// JSpatialEpidemic - A simulation framework to study the spatial aspect of an epidemic.
// JSpatialEpidemic implements a model from "Bailey N.T.J., The Mathematical Theory of 
//     Infectious Diseases and its application - II ed., Charles Griffin & Co. LTD, 1975,
//     pp. 182-188"
// Copyright  2003 Pasquale Cariello, Raul Bagni & Roberto Berchi. 
// See Readme for further details.

import swarm.Globals;
import swarm.space.Grid2d;
import swarm.space.Grid2dImpl;
import swarm.gui.Raster;
import swarm.defobj.Zone;
import swarm.defobj.ZoneImpl;

  
/** 
 * An agent 'live' in a 2D world and is located at a point of a square lattice bounded 
 * by the lines x = +-k, y = +-k.
 * At each step he have a chance to acquire the infection that depends from a parameter p 
 * and from the number of infective in his Moore neighbourhood (eight nearest neighbours) 
 */
 
public class Agent {

  /* *********** */  
  /** PROPERTIES */
  /* *********** */

 
  /*  health state */
  public int state;  /* 0=susceptible; 1=infected; 2=infective; 3=immune */ 
  public int inftime;  /* when the agent became infected
  
  /*  spatial coordinates */
  public int x, y;			

  /* number of the square 
   * the population is divided in sub-gruoups of individuals located along the boundary 
   * of any given square centered on the origin */
  public int square;
  
  /** the world */
  public Grid2d world;				  

  /* the modelSwarm */
  public ModelSwarm model;

  /** the colour (display) */
  public byte agentColor;		



/* *********** */
/**  METHODS   */
/* *********** */

  /* Constructor */
  public Agent (Grid2d w, ModelSwarm mSwarm) { 
    //super(aZone);
    if (world != null) 
      System.err.println ("You can only set the world " +
                          "of an agent at creation time");
    
    world = w;  
    model = mSwarm;
    
    // make sure the user set up world and sector.
    if (world == null) 
      System.err.println ("Agent was created without a world!!");
        
    // The other properties will be set in the model
  }
  
  
  /*
   * Getter/Setter methods
   */
  
  /* get the state of the agent */
  public int getStato () {
    return this.state;
  }
  
  /* set the state of the agent */
  public void setStato (int i) {
    this.state = i;     
  }
  
  /* get the infection time of the agent */
  public int getInfTime () {
    return this.inftime;
  }
  
  /* set the state of the agent */
  public void setInfTime (int i) {
    this.inftime = i;     
  }

  /* get the square of the agent */
  public int getSquare () {
    return this.square;
  }
  
  /* set the square of the agent */
  public void setSquare (int i) {
    this.square = i;     
  }

  /* get the x-coordinate of the agent */
  public int getX () {
    return this.x;
  }
  
  /* get the y-coordinate of the agent */
  public int getY () {
    return this.y;
  }

/**
     This method is a bit dangerous: we blindly put ourselves on top
     of the grid no matter what's underneath us: because Grid2d only
     allows one object per square, we could be destroying data. This
     is poor design, but fortunately doesn't kill us in this
     particular app. If some other object really needed to find all
     objects based on looking in the grid, it would cause
     problems. (But note, here we tell Grid2d to
     turn off its warnings about overwrites) */
  public Agent setX$Y (int inX, int inY) {
    x = inX;
    y = inY;
    world.putObject$atX$Y (this, x, y);		  // yikes!
    return this;
  }
  
  /* get the modelSwarm */
  public ModelSwarm getModel () {
    return model;
  }
  
  /* set the modelSwarm */
  public void setModel (ModelSwarm ms) {
    model = ms;
  }

  /* get the color of the agent */
  public byte getAgentColor() { 
    return agentColor; 
  }

  /* set the color of the agent */
  public void setAgentColor(byte c) { 
    agentColor = c; 
  }
  
  
  /**
     All of the previous code is basic Swarm object programming to manipulate
     the properties of the Object. The real simulation code follows.  
  */
  
  
  /* custom function: the power of a double */  
    public double potenza(double a, int b)  {
      int i;
      double c=1D;
      
      if (b==0) return 1.0;
      else 
        for (i=1; i<=b; i++)
          c=c*a;
         
      return c;
        
    }
    
    /*
    * this method verify if the agent became infected, but it doesn't really set
    * the new state of the agent (see updateAgent()). 
    */
    public void agentStep () {
    double p;  //infection rate by contact
    int lt; //latency time
    double beta; //total infection rate
    int k = model.getK();
    /* neighbours */
    int numinf = 0;  
    int xnow,ynow; 
    Agent anAgent; 
    
    /* initialization */
    p=model.getP();
    lt=model.getLt();
    xnow = this.getX();
    ynow = this.getY();
    
    if (this.getStato() > (byte) 3 || this.getStato() < (byte) 0)  { 
       System.out.println("WARNING: An agent has reached an illegale state!");
    }
    
    if (this.getStato() == (byte) 0)  {  // not yet infected
      for( int i = -1; i <= 1; i++)  //number of infected 8-neighbours
        for( int j = -1; j <= 1; j++)  {
           if ((xnow+i>=0) && (ynow+j>=0) &&
               (xnow+i<world.getSizeX()) && (ynow+j<world.getSizeY()))   {
               anAgent= (Agent) world.getObjectAtX$Y(xnow+i,ynow+j);
               if ((anAgent !=  null) && 
                 (anAgent.getStato()==2)) numinf++;
               //anAgent.drop();
           }
      } 
      beta = 1.0 - potenza((1.0 - p), numinf);  
      if (numinf > 0) {  //at least one neighbours is infective
        if ((Globals.env.uniformDblRand.getDoubleWithMin$withMax (0.0, 1.0)) <= beta) {
              this.setInfTime(Globals.env.getCurrentTime());
        }
      }
    } //end of if (this.getStato() == (byte) 0)

  } //end of AgentStep() 
  
  
  /**
   * this method actually updates the agents' health states (i.e. it is a postStep method).
   * The new situation will be available for the method agentStep at the next tick.
   */
  public void agentUpdate () {
    
    int k = model.getK();
    int lt=model.getLt();
 
    /* update the state and the color of the agent as calculated by agentStep */
    // 1. healthy --> infected
    if (this.getStato() == 0 && this.getInfTime()==Globals.env.getCurrentTime()) {    
      this.setAgentColor((byte) (k+1));
      this.setStato(1);  //infected (not infective)
      //this.getModel().addInfected(); 
    }
    // 2. infected --> infectious
    if (this.getStato() == 1 && 
             this.getInfTime()==(Globals.env.getCurrentTime()-lt)) { //infectious point
      this.setAgentColor((byte) (k+2));
      this.setStato(2);  //infective
      //this.getModel().remInfected();
      }
    // 3. infectious --> immune 
    if (this.getStato() == 2 && 
             this.getInfTime()==(Globals.env.getCurrentTime()-lt-1)) { 
      this.setAgentColor((byte) (k+3));
      this.setStato(3);  //immune
      }  
      
  }
    
    
  /* Insert the object into the raster */ 
  public void drawSelfOn (Raster r) {
    r.drawPointX$Y$Color (x, y, this.getAgentColor());
    //return this;
  }  
    
  
  /**
   * Methods for EZGraph
   */
 
  // Return the number of susceptible agents 
  public double isSusc () {
    if (this.getStato() == 0) return 1.0;
    else return 0.0;                              
  }
  
  // Return the number of NEW infections  
  public double isNewInf () {
    if (this.getInfTime()==Globals.env.getCurrentTime()) return 1.0;
    else return 0.0;                              
  }

  // Return the number of total infected/immune agents 
    public double isInf () {
    if (this.getStato() > 0 && this.getInfTime() > -1) return 1.0;
    else return 0.0;                              
  }
  
  // Return the number of immune agents 
  public double isImmune () {
    if (this.getStato() == 2) return 1.0;
    else return 0.0;                              
  }
  
  // Return the number of infected agents in the square number 0
  public double newInf0 () {
    if (this.getSquare()==0 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  }
  
// Return the number of infected agents in the square number 1
  public double newInf1 () {
    if (this.getSquare()==1 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  } 
  
// Return the number of infected agents in the square number 2
  public double newInf2 () {
    if (this.getSquare()==2 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  }
        
// Return the number of infected agents in the square number 3
  public double newInf3 () {
    if (this.getSquare()==3 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  }

// Return the number of infected agents in the square number 4
  public double newInf4 () {
    if (this.getSquare()==4 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  }

// Return the number of infected agents in the square number 5
  public double newInf5 () {
    if (this.getSquare()==5 && this.getInfTime()==Globals.env.getCurrentTime()) 
       return 1.0;
    else return 0.0;                              
  }


}
