/*
SDX: Documentary System in XML.
Copyright (C) 2000  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.search.lucene.query;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.lucene.search.Hits;

import java.io.IOException;


/*
 *	Les documents qui forment une liste ordonn�e de r�sultats de recherche.
 *
 *	Cette classe permet de g�rer efficacement les tris, car seules les requ�tes
 *	qui demandent un tri autre que la pertinence sont effectivement tri�es.
 */

/**
 * The documents which form an ordered list of search results.
 * This class allows one to manage the sorting effectively, because
 * only the requests which require a sorting other than the relevance
 * are actually sorted.
 */
public class ResultDocuments implements LogEnabled {

    /** Avalon logger to write information. */
    private org.apache.avalon.framework.logger.Logger logger;

    /* The Lucene results, if needed */
    private Hits luceneHits;

    /** Sorted results */
    private ResultDocument[] sortedResults;

    /** The number of results */
    private int nbHits;

    /**Creates a ResultDocuments object
     *
     * <p>
     * A logger must be set and then this object must be setUp.
     *
     * @see #enableLogging
     * @see #setUp
     */
    public ResultDocuments() {
    }

    /**
     *	Builds a list of result documents from Lucene results.
     *
     *	@param	h			The Lucene results.
     */
    public void setUp(Hits h) {
        luceneHits = h;
        nbHits = h.length();
    }

    /**
     *	Builds a list of result documents from an array or sorted ResultDocuments.
     *
     *	@param	sr			The sorted results.
     */
    public void setUp(ResultDocument[] sr) {
        sortedResults = sr;
        nbHits = sr.length;
    }

    /**
     *	Sets the sorted results.
     *
     *	@param	sr			The sorted results.
     */
    public void setSortedResults(ResultDocument[] sr) {
        sortedResults = sr;
        nbHits = sr.length;
        luceneHits = null;
    }

    /**
     *	Returns the number of results.
     */
    public int getLength() {
        return nbHits;
    }

    /**
     *	Returns the relevance ranking of the first document; therefore, the highest ranking is returned.
     */
    public float getMaxScore() throws SDXException, IOException {
        return getDocument(0).getScore();
    }

    /**
     *	Returns a document according to its index in the list.
     *
     *	@param	i	The index of the desired document.
     */
    public ResultDocument getDocument(int i) throws SDXException {
        if (luceneHits != null && i < luceneHits.length()) {
            ResultDocument rDoc = null;
            try {
                if (luceneHits.doc(i) != null) {
                    rDoc = new ResultDocument();
                    rDoc.enableLogging(logger);
                    rDoc.setUp(luceneHits.doc(i), luceneHits.score(i));
                }
            } catch (IOException e) {
                //we could not get the document at the index specified from the hits
                String[] args = new String[1];
                args[0] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_LUCENE_HITS_DOC, args, e);
            } catch (ArrayIndexOutOfBoundsException e) {
                //debugging, why are we getting this from time to time with sdxworld?
                String[] args = new String[1];
                args[0] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_LUCENE_HITS_DOC, args, e);
            }
            return rDoc;
        } else
        //TODO?:should we be checking for null in this case, what if sortedResults is null?
            return sortedResults[i];
    }

    /**
     *	Returns all of the result documents.
     */
    public ResultDocument[] getDocuments() throws SDXException {
        if (luceneHits != null) {
            ResultDocument[] ret = new ResultDocument[getLength()];
            for (int i = 0; i < getLength(); i++) {
                ResultDocument rDoc = new ResultDocument();
                rDoc.enableLogging(logger);
                try {

                    rDoc.setUp(luceneHits.doc(i), luceneHits.score(i));

                } catch (IOException e) {
                    String[] args = new String[1];
                    args[0] = e.getMessage();
                    throw new SDXException(null, SDXExceptionCode.ERROR_GET_LUCENE_HITS_DOC, args, e);
                }

                ret[i] = rDoc;
            }

            return ret;
        } else
        //should we be checking for null in this case?
            return sortedResults;
    }

    /**
     * Sets the logger.
     *
     * @param   logger      The logger.
     */
    public void enableLogging(Logger logger) {
        this.logger = logger;
    }
}
