/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file std_overloads.hpp
 * @date 2007-04-09
 * OPTIONAL HEADER, INCLUDE ONLY IF NEEDED.
 *
 * Overloads the folowing functions:
 * abs, acos, asin, atan, atan2, ceil, cos, cosh, exp, fabs, floor, fmod,
 * frexp, ldexp, log, log10, modf, pow, sin, sinh, sqrt, swap, tan, tanh
 * in the std namespace.
 * Partial std::numeric_limits<Real> specification.
 * Could conflict with simple_functions.hpp,
 * DO NOT USE with simple_functions.hpp.
 */

#ifndef MPFRCPP_EXTRA_STD_OVERLOADS
#define MPFRCPP_EXTRA_STD_OVERLOADS

#include <mpfrcpp/mpfrcpp.hpp>

#include <limits>

namespace std {

    //------------------------------------------------------------

    void swap ( mpfrcpp::Real &x, mpfrcpp::Real &y ) throw() {
        mpfr_swap ( x.getMpfrT(), y.getMpfrT() );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real abs ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Abs ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real acos ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Acos ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real asin ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Asin ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real atan ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Atan ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real atan2 ( const mpfrcpp::Real &x, const mpfrcpp::Real &y )
    throw() {
        return mpfrcpp::Atan2 ( x, y );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real ceil ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Ceil ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real cos ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Cos ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real cosh ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Cosh ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real exp ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Exp ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real fabs ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Abs ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real floor ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Floor ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real fmod ( const mpfrcpp::Real &x, const mpfrcpp::Real &y )
    throw() {
        mpfrcpp::Real n = mpfrcpp::Floor( x / y );
        return x - n*y;
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real frexp ( const mpfrcpp::Real &x, int *exp ) throw() {
        mpfrcpp::Real y ( x );
        *exp = y.getExponent().getMpExpT();
        y.setExponent( mpfrcpp::Exponent( 0 ) );
        return y;
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real ldexp ( const mpfrcpp::Real &x, int exp ) throw() {
        mpfrcpp::Real y;
        mpfr_set_ui_2exp ( y.getMpfrT(), 1, exp,
                           mpfrcpp::Real::getParameters().getDefaultRoundMode().getMpfrRndT() );
        return x*y;
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real log ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Log ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real log10 ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Log10 ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real modf ( const mpfrcpp::Real &x, mpfrcpp::Real *y ) throw() {
        *y = mpfrcpp::Trunc ( x );
        return mpfrcpp::Frac ( x );
    }

    //------------------------------------------------------------

    // x^y

    inline mpfrcpp::Real pow (const mpfrcpp::Real& x,
                              const mpfrcpp::Real& y) throw() {
        return mpfrcpp::Pow (x, y);
    }

    inline mpfrcpp::Real pow (const mpfrcpp::Real& x, int& y) throw() {
        return mpfrcpp::Pow (x, y);
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real sin ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Sin ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real sinh ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Sinh ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real sqrt ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Sqrt ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real tan ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Tan ( x );
    }

    //------------------------------------------------------------

    inline mpfrcpp::Real tanh ( const mpfrcpp::Real &x ) throw() {
        return mpfrcpp::Tanh ( x );
    }

    //------------------------------------------------------------

/**
 * numeric_limits<mpfrcpp::Real>::is_specialized == false since the
 * complete specification depends on type precision.
 * Current code should be used to get epsilon / NaN / Infinity only.
 */

    template<> class numeric_limits<mpfrcpp::Real> {
    public:
        static const bool is_specialized = false; // FIXME
        static const int radix = 2;
        static const bool is_signed = true;
        static const bool is_integer = false;
        static const bool is_iec559 = true;
        static const bool has_infinity = true;
        static const bool has_quiet_NaN = true;
        static const bool is_bounded = true;
        static const bool is_modulo = false;
        static const bool is_exact = false;

        static mpfrcpp::Real epsilon () throw() { return mpfrcpp::Real::epsilon(); }
        static mpfrcpp::Real quiet_NaN() throw() { return mpfrcpp::NaN; }
        static mpfrcpp::Real signaling_NaN() throw() { return mpfrcpp::NaN; }
        static mpfrcpp::Real infinity() throw() { return mpfrcpp::Infinity; }
    };

    //------------------------------------------------------------

} // namespace std

#endif    // MPFRCPP_EXTRA_STD_OVERLOADS
