# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import traceback

import mobius
import pymobius
from . import accounts
from . import autofill
from . import bookmarked_urls
from . import calls
from . import chat_messages
from . import cookies
from . import installed_programs
from . import ip_addresses
from . import opened_files
from . import received_files
from . import sent_files
from . import text_searches
from . import trash_can_entries
from . import turing
from . import visited_urls

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ANT_ID = 'evidence'
ANT_NAME = 'Evidence Finder Agent'
ANT_VERSION = '1.0'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ants for each evidence type
# @deprecated
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ANTS = [
    accounts.Ant,
    autofill.Ant,
    bookmarked_urls.Ant,
    calls.Ant,
    chat_messages.Ant,
    cookies.Ant,
    installed_programs.Ant,
    ip_addresses.Ant,
    opened_files.Ant,
    received_files.Ant,
    sent_files.Ant,
    text_searches.Ant,
    trash_can_entries.Ant,
    visited_urls.Ant,
    turing.Ant,
]


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Bin2Text formatter
# @param value Bytes value
# @return String value
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def formatter_bin2text(value):
    if not value:
        return ''

    elif isinstance(value, str):
        return value

    for encoding in ('utf-8', 'cp1252', 'iso-8859-1'):
        try:
            return value.decode(encoding)
        except UnicodeDecodeError:
            pass

    return '<BINARY> ' + mobius.encoder.hexstring(value)


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Data formatters
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
FORMATTERS = {
    "chat-message-recipients": chat_messages.recipients_formatter,
    "chat-message-text": chat_messages.text_formatter,
    "bin2text": formatter_bin2text,
    "datetime": pymobius.to_string,
    "hexstring": mobius.encoder.hexstring,
    "multiline": lambda lines: '\n'.join(lines or []),
    "string": pymobius.to_string,
}


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generate dict from argument dict
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def args(**kwargs):
    return kwargs


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Getter class for object attribute
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Getter(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, attr_id, attr_format=None):
        self.__attr_id = attr_id
        self.__formatter = FORMATTERS.get(attr_format)

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __call__(self, obj):
        v = getattr(obj, self.__attr_id)

        if self.__formatter:
            return self.__formatter(v)

        return v


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: Evidence
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, item):
        self.id = ANT_ID
        self.name = ANT_NAME
        self.version = ANT_VERSION

        self.__item = item

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Run ant
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def run(self):
        mobius.core.logf(f"INF ant {self.id} started")

        # remove old evidences
        transaction = self.__item.new_transaction()
        self.__item.remove_evidences()
        transaction.commit()

        # run sub-ants
        for ant_class in ANTS:
            ant = ant_class(self.__item)
            mobius.core.logf(f"DBG ant.run started: {ant.name}")

            try:
                ant.run()
            except Exception as e:
                mobius.core.logf(f'WRN {str(e)}\n{traceback.format_exc()}')

            mobius.core.logf(f"DBG ant.run ended: {ant.name}")

        # set ant run
        transaction = self.__item.new_transaction()
        self.__item.set_ant(ANT_ID, ANT_NAME, ANT_VERSION)
        transaction.commit()

        mobius.core.logf(f"INF ant {self.id} ended")


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Evidence model
#
# Each row is composed of dictionaries, with the following keys:
# id: evidence type
# name: evidence name
# description: evidence description
# view_id: (deprecated)
# master_views: list of master views for this evidence type
# detail_views: list of detail views for this evidence type
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
MODEL = [
    args(id="autofill",
         name="Autofill data",
         description="Autofill text data",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='username', is_sortable=True, first_sortable=True),
                      args(id='app', name="Application", is_sortable=True),
                      args(id='fieldname', name="Field Name", is_sortable=True),
                      args(id="value", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='fieldname', name="Field Name"),
                      args(id='value'),
                      args(id='username')
                  ]),
         ]
         ),
    args(id="bookmarked-url",
         name="Bookmarked URLs",
         description="URL's bookmarked by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='creation_time', name="Creation Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="app_name", name="Application", is_sortable=True),
                      args(id="name", is_sortable=True),
                      args(id="url", name="URL", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id="name"),
                      args(id="url", name="URL"),
                      args(id='folder', name="Folder name"),
                      args(id='username', name="User name"),
                      args(id='creation_time', name="Creation Date/time (UTC)"),
                  ]),
         ]
         ),
    args(id="call",
         name="Call Logs",
         description="Regular phone calls and VOIP phone calls metadata",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='source', is_sortable=True),
                      args(id='destination', is_sortable=True, format="multiline"),
                      args(id='duration', is_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="app", name="Application", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime'),
                      args(id='source'),
                      args(id='destination', format="multiline"),
                      args(id='duration'),
                      args(id='username', name="User name"),
                      args(id="app", name="Application"),
                  ]),
         ]),
    args(id="chat-message",
         name="Chat Messages",
         description="Instant chat messages",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='sender', is_sortable=True),
                      args(id='recipients', format="chat-message-recipients", is_sortable=True),
                      args(id="text", format="chat-message-text", is_sortable=True, is_markup=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="app", name="Application", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/time (UTC)"),
                      args(id='search_type', name="Type"),
                      args(id='username', name="User name"),
                      args(id="text", format="chat-message-text"),
                  ]),
         ]
         ),
    args(id="cookie",
         name="Cookies",
         description="HTTP cookies data",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='creation_time', name="Creation Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="app_name", name="Application", is_sortable=True),
                      args(id='domain', is_sortable=True),
                      args(id='name', is_sortable=True),
                      args(id='value', format="bin2text", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='name'),
                      args(id='value', format="bin2text"),
                      args(id='domain'),
                      args(id='creation_time', name="Creation date/time (UTC)", format='datetime'),
                      args(id='last_access_time', name="Last access date/time (UTC)", format='datetime'),
                      args(id='expiration_time', name="Expiration date/time (UTC)", format='datetime'),
                      args(id='last_update_time', name="Last update date/time (UTC)", format='datetime'),
                      args(id='is_deleted'),
                      args(id='is_encrypted'),
                  ]),
         ]
         ),
    args(id="encryption-key",
         name="Encryption Keys",
         description="Encryption keys",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='key_type', name="Type", first_sortable=True),
                      args(id='id', name="ID", is_sortable=True),
                      args(id='value', format="hexstring"),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='key_type', name="Type"),
                      args(id='id', name="ID"),
                      args(id='value', format="hexstring"),
                  ]),
         ]
         ),
    args(id="installed-program",
         name="Installed Programs",
         description="Installed programs",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='name', is_sortable=True),
                      args(id='version'),
                      args(id='description')
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='name'),
                      args(id='version'),
                      args(id='description')
                  ]),
         ]
         ),
    args(id="instant-message",
         name="Instant Messages",
         description="SMS and other instant messages",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id="message_type", name="Type", is_sortable=True),
                      args(id='sender', is_sortable=True),
                      args(id='recipients', format="chat-message-recipients", is_sortable=True),
                      args(id="text"),
                      args(id="app", name="Application", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime'),
                      args(id="message_type", name="Type"),
                      args(id='sender'),
                      args(id='recipients', format="chat-message-recipients"),
                      args(id="text"),
                      args(id="app", name="Application"),
                  ]),
         ]
         ),

    args(id="ip-address",
         name="IP Addresses",
         description="External IP addresses recorded by applications",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='address', name="IP Address", is_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="app_name", name="Application", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='username', name="User name"),
                      args(id='timestamp', name="Date/time (UTC)"),
                      args(id='address', name="IP Address"),
                  ]),
         ]
         ),
    args(id="opened-file",
         name="Opened Files",
         description="Files opened by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id='path', is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='username', name="User name"),
                      args(id='timestamp', name="Date/time (UTC)"),
                      args(id='app_name', name="Application"),
                  ]),
         ]
         ),
    args(id="password-hash",
         name="Password Hashes",
         description="Password hashes, such as NTLM and LM",
         view_id="password-hashes",
         ),
    args(id="password",
         name="Passwords",
         description="Passwords found",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='password_type', name="Type"),
                      args(id='value', name="Password"),
                      args(id='description'),
                  ],
                  exporters=[
                      args(id="wordlist", name="Word list", extensions="txt", function=turing.exporter_wordlist),
                  ])
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='password_type', name="Type"),
                      args(id='value', name="Password"),
                      args(id='description'),
                  ]),
         ]
         ),
    args(id="received-file",
         name="Received Files",
         description="Files received by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/Time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id='filename', name="File name", is_sortable=True),
                      args(id='path', is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Start Time (UTC)"),
                      args(id='username', name="User name"),
                      args(id='app_name', name="Application"),
                      args(id='filename', name="File name"),
                      args(id='path'),
                  ]),
         ]
         ),
    args(id="searched-text",
         name="Searched Texts",
         description="Texts searched by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='search_type', name="Type", is_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="text", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/time (UTC)"),
                      args(id='search_type', name="Type"),
                      args(id='username', name="User name"),
                      args(id="text"),
                  ]),
         ]
         ),
    args(id="sent-file",
         name="Sent Files",
         description="Files sent by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/Time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User account", is_sortable=True),
                      args(id='filename', name="File name", is_sortable=True),
                      args(id='path', is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/Time (UTC)", format='datetime'),
                      args(id='username', name="User account"),
                      args(id='app_name', name="Application"),
                      args(id='filename', name="File name"),
                      args(id='path'),
                  ]),
         ]
         ),
    args(id="trash-can-entry",
         name="Trash Can Entries",
         description="Trash can entries",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='deletion_time', name="Deletion Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='path', is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='path', name="File path"),
                      args(id='size', name="File size"),
                      args(id='deletion_time', name="Deletion Date/time (UTC)"),
                  ]),
         ]
         ),
    args(id="user-account",
         name="User Accounts",
         description="User accounts",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='account_type', name='Type', first_sortable=True),
                      args(id='id', name='ID', is_sortable=True),
                      args(id='password_found'),
                      args(id='password'),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='account_type', name='Type'),
                      args(id='id', name='Account ID'),
                      args(id='name'),
                      args(id='password'),
                      args(id='password_found'),
                  ]),
         ]
         ),
    args(id="visited-url",
         name="Visited URLs",
         description="URLs visited by users",
         master_views=[
             args(id="table",
                  columns=[
                      args(id='timestamp', name="Date/time (UTC)", format='datetime', first_sortable=True),
                      args(id='username', name="User name", is_sortable=True),
                      args(id="url", name="URL", is_sortable=True),
                  ]),
         ],
         detail_views=[
             args(id="metadata",
                  rows=[
                      args(id='timestamp', name="Date/time (UTC)"),
                      args(id="url", name="URL"),
                      args(id='title', name="Page title"),
                      args(id='username', name="User name"),
                  ]),
         ]
         ),
]
