# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2018 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius.p2p
from pymobius.p2p.local_file import *
import decoder_share_metadata
import mobius
import struct
import binascii

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Entry
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Entry (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.username = None
    self.sharel_path = None
    self.path = None
    self.size = None
    self.title = None
    self.artist = None
    self.album = None
    self.category = None
    self.year = None
    self.vidinfo = None
    self.language = None
    self.url = None
    self.comment = None
    self.download_completed_time = None
    self.flag_shared = True
    self.flag_corrupted = False
    self.flag_completed = False

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief decrypt data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decrypt (data, seed):
  out = ''
  for c in data:
    out += chr (ord (c) ^ (seed >> 8))
    seed = ((ord (c) + seed) * 23219 + 36126) & 0xffff
  return out

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief get file type according to amime field
# @see const_ares.pas and helper_mimetypes.pas
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def get_file_type (mime):
  return {
      1 : 'Audio',
      2 : 'Audio',
      3 : 'Software',
      4 : 'Audio',
      5 : 'Video',
      6 : 'Document',
      7 : 'Image'}.get (mime, 'Other')

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve data from ShareL.dat
# @see get_cached_metas (helper_library_db.pas)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (model, f, username):
  my_shared_folder_path = f.path[:-15].replace ('/', '\\') + 'My Shared Folder\\'

  # try to open file
  if not f:
    return

  reader = f.new_reader ()
  if not reader:
    return

  # decode header
  decoder = mobius.decoder.data_decoder (reader)
  signature = decoder.get_string_by_size (14)
  
  if signature != '__ARESDB1.04L_':
    return

  # add data to model
  for entry in retrieve_entries (decoder):
    entry.username = username
    entry.sharel_path = f.path

    lf = new_local_file (entry)
    model.local_files.append (lf)

  # set file handled
  pymobius.p2p.set_handled (model.item, f)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve entries from file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve_entries (decoder):
  entries = []
  pos = decoder.tell ()
  size = decoder.get_size ()
  
  while pos < size:

    # entry header
    enc_data = decoder.get_bytearray_by_size (47)
    pos += 47
    data = decrypt (enc_data, 13871)

    hash_sha1, mime, e_size, param1, param2, param3, data_size = struct.unpack ('<20sB4xQIIIH', data)

    entry = Entry ()
    entry.mime = mime
    entry.size = e_size
    entry.hash_sha1 = binascii.hexlify (hash_sha1)

    # entry metadata    
    if data_size > 0:
      enc_data = decoder.get_bytearray_by_size (data_size)
      data = decrypt (enc_data, 13872)
      pos += data_size

      metadata = decoder_share_metadata.decode (data)
      for var, value in metadata.iteritems ():
        setattr (entry, var, value)

      # entry is downloaded if download completed time is set
      entry.flag_downloaded = bool (entry.download_completed_time)

      # entry is complete @todo check when
      entry.flag_completed = False

    # add entry
    entries.append (entry)

  return entries

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Create local file
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def new_local_file (entry):
  f = local_file ()
  f.username = entry.username
  f.app_id = 'ares'
  f.app = 'Ares Galaxy'
  f.path = entry.path
  f.size = entry.size

  # name
  if '\\' in f.path:
    f.name = f.path.rsplit ('\\', 1)[1]
  else:
    f.name = f.path

  # flags
  f.flag_shared = STATE_YES
  f.flag_downloaded = STATE_YES if entry.flag_downloaded else STATE_NO
  # @todo f.flag_uploaded
  f.flag_corrupted = STATE_YES if entry.flag_corrupted else STATE_NO
  # @todo f.flag_completed

  # hashes
  f.set_hash ('sha1', entry.hash_sha1)

  # metadata
  f.add_metadata ('File type', get_file_type (entry.mime))
  f.add_metadata ('Title', entry.title)
  f.add_metadata ('Artist', entry.artist)
  f.add_metadata ('Album', entry.album)
  f.add_metadata ('Category', entry.category)
  f.add_metadata ('Year', entry.year)
  f.add_metadata ('Video Info', entry.vidinfo)
  f.add_metadata ('Language', entry.language)
  f.add_metadata ('URL', entry.url)
  f.add_metadata ('Comment', entry.comment)
  f.add_metadata ('ShareL.dat path', entry.sharel_path)

  return f
