// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "data_decoder.h"
#include <mobius/datetime/datetime.h>
#include <mobius/datetime/timedelta.h>
#include <mobius/datetime/conv_nt_timestamp.h>
#include <mobius/datetime/conv_unix_timestamp.h>
#include <mobius/io/bytearray_io.h>
#include <mobius/charset.h>
#include <config.h>
#include <cstdio>

namespace mobius
{
namespace decoder
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param in reader object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_decoder::data_decoder (mobius::io::reader in)
  : in_ (in)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param data bytearray object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_decoder::data_decoder (const mobius::bytearray& data)
  : in_ (mobius::io::new_bytearray_reader (data))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief skip n bytes
//! \param size size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_decoder::skip (size_type size)
{
  in_.skip (size);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief jump to position pos
//! \param pos position from the start of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
data_decoder::seek (size_type pos)
{
  in_.seek (pos);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get current position
//! \return position from the start of data
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
data_decoder::size_type
data_decoder::tell () const
{
  return in_.tell ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an int8
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int8_t
data_decoder::get_int8 ()
{
  return std::int8_t (get_uint8 ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an int16 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int16_t
data_decoder::get_int16_le ()
{
  return std::int16_t (get_uint16_le ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an int32 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
int32_t
data_decoder::get_int32_le ()
{
  return std::int32_t (get_uint32_le ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint8
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint8_t
data_decoder::get_uint8 ()
{
  const bytearray data = in_.read (1);

  return data[0];
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint16 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint16_t
data_decoder::get_uint16_le ()
{
  const bytearray data = in_.read (2);
  const uint8_t *p = data.begin ();

  return std::uint16_t (*p) | (std::uint16_t (*(p + 1)) << 8);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint16 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint16_t
data_decoder::get_uint16_be ()
{
  const bytearray data = in_.read (2);
  const uint8_t *p = data.begin ();

  return (std::uint16_t (*p) << 8) | std::uint16_t (*(p + 1));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint32 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint32_t
data_decoder::get_uint32_le ()
{
  const bytearray data (in_.read (4));
  const uint8_t *p = data.begin ();

  return std::uint32_t (*p) |
         (std::uint32_t (*(p + 1)) << 8) |
         (std::uint32_t (*(p + 2)) << 16) |
         (std::uint32_t (*(p + 3)) << 24);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint32 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint32_t
data_decoder::get_uint32_be ()
{
  const bytearray data (in_.read (4));
  const uint8_t *p = data.begin ();

  return std::uint32_t (*(p + 3)) |
         (std::uint32_t (*(p + 2)) << 8) |
         (std::uint32_t (*(p + 1)) << 16) |
         (std::uint32_t (*p) << 24);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint64 (le)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint64_t
data_decoder::get_uint64_le ()
{
  const bytearray data (in_.read (8));
  const uint8_t *p = data.begin ();

  return std::uint64_t (*p) |
         (std::uint64_t (*(p + 1)) << 8) |
         (std::uint64_t (*(p + 2)) << 16) |
         (std::uint64_t (*(p + 3)) << 24) |
         (std::uint64_t (*(p + 4)) << 32) |
         (std::uint64_t (*(p + 5)) << 40) |
         (std::uint64_t (*(p + 6)) << 48) |
         (std::uint64_t (*(p + 7)) << 56);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode an uint64 (be)
//! \return value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
uint64_t
data_decoder::get_uint64_be ()
{
  const bytearray data (in_.read (8));
  const uint8_t *p = data.begin ();

  return std::uint64_t (*(p + 7)) |
         (std::uint64_t (*(p + 6)) << 8) |
         (std::uint64_t (*(p + 5)) << 16) |
         (std::uint64_t (*(p + 4)) << 24) |
         (std::uint64_t (*(p + 3)) << 32) |
         (std::uint64_t (*(p + 2)) << 40) |
         (std::uint64_t (*(p + 1)) << 48) |
         (std::uint64_t (*p) << 56);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode HFS timestamp
//! \return datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
data_decoder::get_hfs_datetime ()
{
  mobius::datetime::datetime dt;
  auto timestamp = get_uint32_be ();

  if (timestamp)
    dt = mobius::datetime::datetime (1904, 1, 1, 0, 0, 0) + mobius::datetime::timedelta (0, 0, timestamp, 0);

  return dt;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode ISO9660 timestamp
//! \return datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
data_decoder::get_iso9660_datetime ()
{
  auto data = in_.read (17);

  // parse date/time from string
  int y, m, d;
  int hh, mm, ss;
  sscanf (reinterpret_cast<const char *> (data.data ()),
          "%04d%02d%02d%02d%02d%02d",
          &y, &m, &d, &hh, &mm, &ss);

  // create datetime object
  mobius::datetime::datetime dt;

  if (y || m || d || hh || mm || ss)
    {
      dt = mobius::datetime::datetime (y, m, d, hh, mm, ss);

      // offset from UTC in 15min intervals
      int offset = static_cast<char> (data[16]);

      if (offset)
        {
          mobius::datetime::timedelta delta (0, 0, offset * 15 * 60, 0);
          dt = dt + delta;  //dt += delta;
        }
    }

  return dt;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode NT timestamp
//! \return datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
data_decoder::get_nt_datetime ()
{
  mobius::datetime::datetime dt;
  std::uint64_t timestamp = get_uint64_le ();

  if (timestamp)
    dt = mobius::datetime::datetime_from_nt_timestamp (timestamp);

  return dt;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode UNIX timestamp
//! \return datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
data_decoder::get_unix_datetime ()
{
  mobius::datetime::datetime dt;
  auto timestamp = get_uint32_le ();

  if (timestamp)
    dt = mobius::datetime::datetime_from_unix_timestamp (timestamp);

  return dt;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode bytearray by size
//! \param size size in bytes
//! \return bytearray
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
data_decoder::get_bytearray_by_size (std::size_t size)
{
  return in_.read (size);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode string by size
//! \param size size in bytes
//! \return string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
data_decoder::get_string_by_size (std::size_t size, const std::string& encoding)
{
  bytearray data = in_.read (size);
  std::string result;

  if (encoding == "ASCII" || encoding == "UTF-8")
    result = std::string (data.begin (), data.end ());

  else
    result = conv_charset_to_utf8 (data, encoding);

  auto pos = result.find ('\0');
  if (pos != std::string::npos)
    result.erase (pos);

  return result;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode GUID
//! \return GUID as formatted string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
data_decoder::get_guid ()
{
  uint32_t guid1 = get_uint32_le ();
  uint16_t guid2 = get_uint16_le ();
  uint16_t guid3 = get_uint16_le ();
  uint16_t guid4 = get_uint16_be ();
  bytearray guid5 = in_.read (6);

  char buffer[64];
  sprintf (buffer, "%08x-%04x-%04x-%04x-%02x%02x%02x%02x%02x%02x",
           guid1, guid2, guid3, guid4,
           guid5[0],
           guid5[1],
           guid5[2],
           guid5[3],
           guid5[4],
           guid5[5]);

  return buffer;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode UUID
//! \return UUID as formatted string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
data_decoder::get_uuid ()
{
  std::string uuid = in_.read (16).to_hexstring ();

  return uuid.substr (0, 8) + '-' +
         uuid.substr (8, 4) + '-' +
         uuid.substr (12, 4) + '-' +
         uuid.substr (16, 4) + '-' +
         uuid.substr (20);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief decode IPv4
//! \return IPv4 as formatted string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
data_decoder::get_ipv4 ()
{
  bytearray data = in_.read (4);

  char buffer[64];
  sprintf (buffer, "%d.%d.%d.%d", data[0], data[1], data[2], data[3]);

  return buffer;
}

} // namespace decoder
} // namespace mobius
