// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "ui_impl_base.h"
#include <mobius/exception.inc>
#include <map>
#include <memory>
#include <stdexcept>

namespace mobius
{
namespace ui
{
namespace
{
// Global UI implementation
static std::shared_ptr <ui_impl_base> g_impl_;

// Implementations: ID -> (description, builder function)
std::map <std::string, std::pair <std::string, std::function <std::shared_ptr <ui_impl_base> ()>>> implementations_;
} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Add UI implementation class
//! \param id UI implementation ID
//! \param description UI implementation description
//! \param f UI implementation builder function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
add_ui_implementation (
  const std::string& id,
  const std::string& description,
  std::function <std::shared_ptr <ui_impl_base> ()> f)
{
  implementations_[id] = std::make_pair (description, f);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove UI implementation class
//! \param id UI implementation ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
remove_ui_implementation (const std::string& id)
{
  implementations_.erase (id);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get list of UI implementation classes
//! \return List of implementation classes ID and description
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::list <std::pair <std::string, std::string>>
get_ui_implementations ()
{
  std::list <std::pair <std::string, std::string>> implementations;
  
  for (auto& i : implementations_)
    implementations.push_back (std::make_pair (i.first, i.second.first));
  
  return implementations;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set current UI implementation class
//! \param id UI implementation ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
set_ui_implementation (const std::string& id)
{
  if (g_impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("UI implementation already set"));

  // search implementation
  auto iter = implementations_.find (id);

  if (iter != implementations_.end ())
    g_impl_ = iter->second.second ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get current UI implementation class
//! \return Smart pointer to UI implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::shared_ptr <ui_impl_base>
get_ui_implementation ()
{
  // if g_impl_ is not set, create using first UI implementation available
  if (!g_impl_)
    {
      if (implementations_.size () == 0)
        throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("no UI implementation found"));

      else
        g_impl_ = implementations_.begin ()->second.second ();
    }

  // return implementation instance
  return g_impl_;
}

} // namespace ui
} // namespace mobius
