#ifndef MOBIUS_CRYPT_CIPHER_STREAM_H
#define MOBIUS_CRYPT_CIPHER_STREAM_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <cstdint>
#include <mobius/crypt/cipher_base.h>

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Abstract base class for stream ciphers
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class cipher_stream : public cipher_base
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get block size
  //! \return Block size in bytes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::size_t
  get_block_size () const noexcept override
  {
    return 0;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Encrypt data block
  //! \param first input iterator to the first element
  //! \param last input iterator to the element after the last one
  //! \param result output iterator
  //! \return output iterator updated
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename InputIterator, typename OutputIterator> OutputIterator
  encrypt (InputIterator first, InputIterator last, OutputIterator result)
  {
    while (first != last)
      {
        *result = _encrypt_byte (*first);
        ++first;
        ++result;
      }

    return result;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Decrypt data block
  //! \param first input iterator to the first element
  //! \param last input iterator to the element after the last one
  //! \param result output iterator
  //! \return output iterator updated
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename InputIterator, typename OutputIterator> OutputIterator
  decrypt (InputIterator first, InputIterator last, OutputIterator result)
  {
    while (first != last)
      {
        *result = _decrypt_byte (*first);
        ++first;
        ++result;
      }

    return result;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Encrypt a bytearray
  //! \param data bytearray
  //! \return encrypted bytearray
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  encrypt (const mobius::bytearray& data) override
  {
    mobius::bytearray buffer (data.size ());
    encrypt (data.begin (), data.end (), buffer.begin ());
    return buffer;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Decrypt a bytearray
  //! \param data bytearray
  //! \return decrypted bytearray
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  decrypt (const mobius::bytearray& data) override
  {
    mobius::bytearray buffer (data.size ());
    decrypt (data.begin (), data.end (), buffer.begin ());
    return buffer;
  }

protected:
  virtual std::uint8_t _encrypt_byte (std::uint8_t) noexcept = 0;
  virtual std::uint8_t _decrypt_byte (std::uint8_t) noexcept = 0;
};

} // namespace crypt
} // namespace mobius

#endif
