# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius
import pymobius.app.internet_explorer

APP_ID = 'ie'
APP_NAME = 'Internet Explorer'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve Internet Explorer browser artifacts
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (os, item, model):

  # retrieve data from OS installed
  app_model = pymobius.app.internet_explorer.model (item)

  # add profiles
  for profile in app_model.get_profiles ():
    p = pymobius.Data ()
    p.app = APP_ID
    p.app_name = APP_NAME
    p.name = profile.name
    p.path = profile.path
    p.username = profile.username
    
    model.profiles.append (p)

    # add URL history
    history = [ e for e in profile.get_history () if not e.url.startswith ('file:///') and not e.url.startswith (':Host: ') ]

    for entry in history:
      h = pymobius.Data ()
      h.app = APP_ID
      h.app_name = APP_NAME
      h.url = entry.url
      h.username = profile.username
      h.itemname = item.name
      h.profile_id = profile.name
      h.profile_path = profile.path

      # set timestamp according to history file type
      if entry.file_type == 'weekly':
        h.timestamp = str (entry.timestamp)[:19] + ' (local time)'

      else:  
        h.timestamp = str (entry.timestamp)[:19]

      # set metadata
      h.metadata = []
      h.metadata.append (('Page title', entry.title or ''))
      h.metadata.append (('index.dat file path', entry.index_dat_path))
      h.metadata.append (('index.dat creation time (UTC)', entry.index_dat_creation_time))
      h.metadata.append (('index.dat last modification time (UTC)', entry.index_dat_last_modification_time))
      h.metadata.append (('index.dat offset', '0x%08x' % entry.offset))
      h.metadata.append (('Record type', entry.type))
      h.metadata.append (('Record size', entry.size))
      h.metadata.append (('Record primary time (UTC)', entry.primary_time))

      if entry.file_type in ('daily', 'weekly'):
        h.metadata.append (('Record secondary time (local time)', entry.secondary_time))
      else:
        h.metadata.append (('Record secondary time (UTC)', entry.secondary_time))

      local_time = entry.tags.get (0x18)
      if local_time:
        h.metadata.append (('Local time (tag 0x18)', local_time))

      if entry.type == 'URL':
        h.metadata.append (('Expiration time', entry.expiration_time))
        h.metadata.append (('Last sync time', entry.last_sync_time))
        h.metadata.append (('Hits', entry.hits))
        h.metadata.append (('Cached file path', entry.filename))

      model.history.append (h)

    # add form history
    for entry in profile.get_form_history ():
      h = pymobius.Data ()
      h.app = APP_ID
      h.app_name = APP_NAME
      h.username = profile.username
      h.fieldname = entry.fieldname
      h.itemname = item.name
      h.value = entry.value
      h.profile_id = profile.name
      h.profile_path = profile.path

      h.metadata = []
      h.metadata.append (('First used date/time', str (entry.first_used_time)[:19]))
      h.metadata.append (('Last used date/time', str (entry.last_used_time)[:19]))

      model.form_history.append (h)
