// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "credhist_entry.h"
#include "cipher_info.h"
#include "hash_info.h"
#include <mobius/charset.h>
#include <mobius/crypt/cipher.h>
#include <mobius/crypt/hash_sha1.h>
#include <mobius/crypt/hmac.h>
#include <mobius/decoder/data_decoder.h>
#include <mobius/exception.inc>
#include <mobius/turing/pbkdf2_hmac_ms.h>
#include <stdexcept>

namespace mobius
{
namespace turing
{
namespace dpapi
{

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief credhist_entry implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class credhist_entry::impl
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl (const impl&) = delete;
  impl (impl&&) = delete;
  explicit impl (mobius::io::reader);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Operators
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl& operator= (const impl&) = delete;
  impl& operator= (impl&&) = delete;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get revision
  //! \return Revision
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_revision () const
  {
    return revision_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get guid
  //! \return Guid
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_guid () const
  {
    return guid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get size
  //! \return Size
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_size () const
  {
    return size_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get type
  //! \return Type
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_type () const
  {
    return type_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get hash id
  //! \return Hash id
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_hash_id () const
  {
    return hash_id_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get iterations
  //! \return Iterations
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_iterations () const
  {
    return iterations_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get cipher id
  //! \return Cipher id
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_cipher_id () const
  {
    return cipher_id_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get salt
  //! \return Salt
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  get_salt () const
  {
    return salt_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get sid
  //! \return Sid
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string
  get_sid () const
  {
    return sid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get hash sha1
  //! \return Hash sha1
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  get_hash_sha1 () const
  {
    return hash_sha1_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get hash md4
  //! \return Hash md4
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  mobius::bytearray
  get_hash_md4 () const
  {
    return hash_md4_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  bool decrypt_with_key (const mobius::bytearray&);
  bool decrypt_with_password_hash (const mobius::bytearray&);
  bool decrypt_with_password (const std::string&);

private:
  //! \brief Struct revision
  std::uint32_t revision_;

  //! \brief GUID
  std::string guid_;

  //! \brief Entry size in bytes
  std::uint32_t size_;

  //! \brief Type
  std::uint32_t type_ = 0;

  //! \brief Hash algorithm ID
  std::uint32_t hash_id_ = 0;

  //! \brief Number of key iterations
  std::uint32_t iterations_ = 0;

  //! \brief Cipher algorithm ID
  std::uint32_t cipher_id_ = 0;

  //! \brief Salt
  mobius::bytearray salt_;

  //! \brief SID
  std::string sid_;

  //! \brief SHA1 hash value
  mobius::bytearray hash_sha1_;

  //! \brief MD4 hash value
  mobius::bytearray hash_md4_;

  //! \brief SHA1 hash length
  std::uint32_t sha1_length_ = 0;

  //! \brief MD4 hash length
  std::uint32_t md4_length_ = 0;

  //! \brief Cipher text
  mobius::bytearray cipher_text_;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param reader Reader object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
credhist_entry::impl::impl (mobius::io::reader reader)
{
  constexpr int FOOTER_SIZE = 24;

  // check if there are enough bytes to read
  mobius::decoder::data_decoder decoder (reader);

  if (decoder.tell () < FOOTER_SIZE)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Not enough bytes to read"));

  // read entry footer
  decoder.seek (decoder.tell () - FOOTER_SIZE);
  auto pos = decoder.tell ();

  revision_ = decoder.get_uint32_le ();
  guid_ = decoder.get_guid ();
  size_ = decoder.get_uint32_le ();

  // read entry data, if available
  if (size_)
    {
      decoder.seek (decoder.tell () - size_);

      type_ = decoder.get_uint32_le ();
      hash_id_ = decoder.get_uint32_le ();
      iterations_ = decoder.get_uint32_le ();
      auto sid_size = decoder.get_uint32_le ();
      cipher_id_ = decoder.get_uint32_le ();
      sha1_length_ = decoder.get_uint32_le ();
      md4_length_ = decoder.get_uint32_le ();
      salt_ = decoder.get_bytearray_by_size (16);

      // read user SID, if available
      if (sid_size > 0)
        sid_ = decoder.get_sid ();

      // read cipher text
      cipher_text_ = decoder.get_bytearray_by_size (pos - reader.tell ());
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt entry with key
//! \param key Decryption key
//! \return true if entry has been decrypted
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::impl::decrypt_with_key (const mobius::bytearray& key)
{
  // generate prekey
  auto hash_id = mobius::turing::dpapi::get_hash_id (hash_id_);
  auto cipher_key_length = get_cipher_key_length (cipher_id_);
  auto cipher_salt_length = get_cipher_salt_length (cipher_id_);
  auto value = mobius::turing::pbkdf2_hmac_ms (key, salt_, iterations_, cipher_key_length + cipher_salt_length, hash_id);
  
  auto prekey = value.slice (0, cipher_key_length - 1);
  auto presalt = value.slice (cipher_key_length, cipher_key_length + cipher_salt_length - 1);

  // decrypt cipher text
  auto cipher_id = mobius::turing::dpapi::get_cipher_id (cipher_id_);
  mobius::crypt::cipher cipher (cipher_id, prekey, "cbc", presalt);
  auto cleartxt = cipher.decrypt (cipher_text_);

  // check decryption
  bool flag_decrypted = true;
  constexpr std::uint32_t NT_HASH_LENGTH = 16;
  std::uint32_t md4_real_length = std::min (NT_HASH_LENGTH, md4_length_);  

  if (md4_length_ > md4_real_length)
    {
      auto remain = cleartxt.slice (sha1_length_ + md4_real_length, sha1_length_ + md4_length_ - 1);
      flag_decrypted = remain.all_equal (0);
    }

  // fill hashes
  if (flag_decrypted)
    {
      hash_sha1_ = cleartxt.slice (0, sha1_length_ - 1);
      hash_md4_ = cleartxt.slice (sha1_length_, sha1_length_ + md4_real_length - 1);
    }

  return flag_decrypted;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt entry with password hash
//! \param h Password hash
//! \return true if entry has been decrypted
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::impl::decrypt_with_password_hash (const mobius::bytearray& h)
{
  mobius::crypt::hmac hmac (h, "sha1");
  hmac.update (conv_charset (bytearray (sid_) + bytearray ({0}), "ASCII", "UTF-16LE"));

  return decrypt_with_key (hmac.get_digest ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt entry with password
//! \param password Password
//! \return true if entry has been decrypted
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::impl::decrypt_with_password (const std::string& password)
{
  mobius::crypt::hash_sha1 h;
  h.update (conv_charset (password, "UTF-8", "UTF-16LE"));

  return decrypt_with_password_hash (h.get_digest ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param reader Reader object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
credhist_entry::credhist_entry (mobius::io::reader reader)
  : impl_ (std::make_shared <impl> (reader))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get revision
//! \return Revision
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_revision () const
{
  return impl_->get_revision ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get guid
//! \return Guid
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
credhist_entry::get_guid () const
{
  return impl_->get_guid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get size
//! \return Size
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_size () const
{
  return impl_->get_size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get type
//! \return Type
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_type () const
{
  return impl_->get_type ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get hash id
//! \return Hash id
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_hash_id () const
{
  return impl_->get_hash_id ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get iterations
//! \return Iterations
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_iterations () const
{
  return impl_->get_iterations ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get cipher id
//! \return Cipher id
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
credhist_entry::get_cipher_id () const
{
  return impl_->get_cipher_id ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get salt
//! \return Salt
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
credhist_entry::get_salt () const
{
  return impl_->get_salt ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get sid
//! \return Sid
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
credhist_entry::get_sid () const
{
  return impl_->get_sid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get hash sha1
//! \return Hash sha1
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
credhist_entry::get_hash_sha1 () const
{
  return impl_->get_hash_sha1 ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get hash md4
//! \return Hash md4
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
credhist_entry::get_hash_md4 () const
{
  return impl_->get_hash_md4 ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt cipher text using key
//! \param key Key
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::decrypt_with_key (const mobius::bytearray& key)
{
  return impl_->decrypt_with_key (key);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt cipher text using password hash
//! \param h Password hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::decrypt_with_password_hash (const mobius::bytearray& h)
{
  return impl_->decrypt_with_password_hash (h);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Decrypt cipher text using password
//! \param password Password encoded in UTF-8
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
credhist_entry::decrypt_with_password (const std::string& password)
{
  return impl_->decrypt_with_password (password);
}

} // namespace dpapi
} // namespace turing
} // namespace mobius
