// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file C++ API <i>mobius.crypt</i> module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "cipher_3des.h"
#include "cipher_aes.h"
#include "cipher_blowfish.h"
#include "cipher_des.h"
#include "cipher_rc4.h"
#include "cipher_rot13.h"
#include "cipher_zip.h"
#include "hash.h"
#include "hmac.h"
#include "api.h"
#include <mobius/crypt/pkcs5.h>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief pbkdf1 function implementation
//! \param self function object
//! \param args argument list
//! \return Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_pbkdf1 (PyObject *, PyObject *args)
{
  // parse input args
  const char *arg_password;
  const char *arg_salt;
  unsigned long arg_iterations = 0;
  unsigned int arg_dklen = 0;
  const char *arg_hash_id = "sha1";

  if (!PyArg_ParseTuple (args, "ssLI|s", &arg_password, &arg_salt, &arg_iterations, &arg_dklen, &arg_hash_id))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyBytes_from_bytearray (
              mobius::crypt::pbkdf1 (arg_password, arg_salt, arg_iterations, arg_dklen, arg_hash_id)
            );
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief pbkdf2_hmac function implementation
//! \param self function object
//! \param args argument list
//! \return Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
func_pbkdf2_hmac (PyObject *, PyObject *args)
{
  // parse input args
  const char *arg_password;
  const char *arg_salt;
  unsigned long arg_iterations = 0;
  unsigned int arg_dklen = 0;
  const char *arg_hash_id = "sha1";

  if (!PyArg_ParseTuple (args, "ssLI|s", &arg_password, &arg_salt, &arg_iterations, &arg_dklen, &arg_hash_id))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyBytes_from_bytearray (
              mobius::crypt::pbkdf2_hmac (arg_password, arg_salt, arg_iterations, arg_dklen, arg_hash_id)
            );
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef module_methods[] =
{
  {
    "pbkdf1",
    func_pbkdf1,
    METH_VARARGS,
    "Generate derived key using PBKDF1"
  },
  {
    "pbkdf2_hmac",
    func_pbkdf2_hmac,
    METH_VARARGS,
    "Generate derived key using PBKDF2_HMAC"
  },
  {
    NULL,
    NULL,
    0,
    NULL
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"cipher_3des", &crypt_cipher_3des_t},
  {"cipher_aes", &crypt_cipher_aes_t},
  {"cipher_blowfish", &crypt_cipher_blowfish_t},
  {"cipher_des", &crypt_cipher_des_t},
  {"cipher_rc4", &crypt_cipher_rc4_t},
  {"cipher_rot13", &crypt_cipher_rot13_t},
  {"cipher_zip", &crypt_cipher_zip_t},
  {"hash", &crypt_hash_t},
  {"hmac", &crypt_hmac_t},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create mobius.crypt module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
new_crypt_module ()
{
  // initialize module mobius.crypt
  PyObject* module = Py_InitModule3 (
                       "mobius.crypt",
                       module_methods,
                       "Mobius Forensic Toolkit mobius.crypt module"
                     );

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // return module
  return module;
}
