// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "api_writer.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_writer_tp_dealloc (api_writer_o *);
static PyObject *api_writer_get_is_seekable (api_writer_o *);
static PyObject *api_writer_get_is_rewindable (api_writer_o *);
static PyObject *api_writer_f_write (api_writer_o *, PyObject *);
static PyObject *api_writer_f_tell (api_writer_o *, PyObject *);
static PyObject *api_writer_f_seek (api_writer_o *, PyObject *);
static PyObject *api_writer_f_rewind (api_writer_o *, PyObject *);
static PyObject *api_writer_f_skip (api_writer_o *, PyObject *);
static PyObject *api_writer_f_flush (api_writer_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_writer_getsetters[] =
{
  {
    (char *) "is_seekable",
    (getter) api_writer_get_is_seekable,
    (setter) 0,
    (char *) "check if writer is seekable", NULL
  },
  {
    (char *) "is_rewindable",
    (getter) api_writer_get_is_rewindable,
    (setter) 0,
    (char *) "check if writer is rewindable", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef api_writer_methods[] =
{
  {
    (char *) "write",
    (PyCFunction) api_writer_f_write,
    METH_VARARGS,
    "writes bytes to writer"
  },
  {
    (char *) "tell",
    (PyCFunction) api_writer_f_tell,
    METH_VARARGS,
    "get current write position"
  },
  {
    (char *) "seek",
    (PyCFunction) api_writer_f_seek,
    METH_VARARGS,
    "set current write position"
  },
  {
    (char *) "rewind",
    (PyCFunction) api_writer_f_rewind,
    METH_VARARGS,
    "set current write position to the beginning of data"
  },
  {
    (char *) "skip",
    (PyCFunction) api_writer_f_skip,
    METH_VARARGS,
    "set write position n bytes ahead"
  },
  {
    (char *) "flush",
    (PyCFunction) api_writer_f_flush,
    METH_VARARGS,
    "write down data"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_writer_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "api.writer",                             		// tp_name
  sizeof (api_writer_o),                    		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) api_writer_tp_dealloc,       		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "writer for mobius.io.file",             		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  api_writer_methods,                       		// tp_methods
  0,                                       		// tp_members
  api_writer_getsetters,                    		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  0                                        		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: tp_alloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
api_writer_o *
api_writer_tp_alloc ()
{
  return (api_writer_o *) api_writer_t.tp_alloc (&api_writer_t, 0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_writer_tp_dealloc (api_writer_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: is_seekable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_get_is_seekable (api_writer_o *self)
{
  return PyBool_FromLong (self->obj->is_seekable ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: is_rewindable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_get_is_rewindable (api_writer_o *self)
{
  return PyBool_FromLong (self->obj->is_rewindable ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: write
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_write (api_writer_o *self, PyObject *args)
{
  // parse input args
  const std::uint8_t *arg_data_buffer;
  int arg_data_size;

  if (!PyArg_ParseTuple (args, "s#", &arg_data_buffer, &arg_data_size))
    return NULL;

  // execute C++ function
  try
    {
      self->obj->write (mobius::bytearray (arg_data_buffer, arg_data_size));
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: tell
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_tell (api_writer_o *self, PyObject *args)
{
  return PyInt_FromLong (self->obj->tell ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: seek
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_seek (api_writer_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_offset;
  int arg_whence = 0;

  if (!PyArg_ParseTuple (args, "L|i", &arg_offset, &arg_whence))
    return NULL;

  // execute C++ function
  mobius::io::writer::whence_type w;

  if (arg_whence == 0)
    w = mobius::io::writer::whence_type::beginning;

  else if (arg_whence == 1)
    w = mobius::io::writer::whence_type::current;

  else if (arg_whence == 2)
    w = mobius::io::writer::whence_type::end;

  self->obj->seek (arg_offset, w);

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: rewind
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_rewind (api_writer_o *self, PyObject *args)
{
  self->obj->rewind ();

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: skip
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_skip (api_writer_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_size;

  if (!PyArg_ParseTuple (args, "L", &arg_size))
    return NULL;

  // execute C++ function
  self->obj->skip (arg_size);

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.writer: flush
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_writer_f_flush (api_writer_o *self, PyObject *args)
{
  self->obj->flush ();

  Py_INCREF (Py_None);
  return Py_None;
}
