# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
METHOD_NEW_PART, METHOD_MERGE, METHOD_REPLACE = range (3)
XML_ENCODING='utf-8'

import libxml2
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Part
# =i=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Part (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.category = None
    self.id = None
    self.attributes = {}

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @deprecated Persistence layer for Part Catalogue prior to Mobius 0.4.4
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load parts.xml file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    parts = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      parts = self.load_parts (node)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <parts>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_parts (self, node):
    parts = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'part':
        part = self.load_part (node)
        parts.append (part)

      node = node.next

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <part>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_part (self, node):
    part = Part ()
    part.category = self.__get_prop (node, 'category')
    part.id = self.__get_prop (node, 'number')
    part.attributes = {}

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        id, value = self.load_attribute (node)
        part.attributes[id] = value

      node = node.next

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    id = self.__get_prop (node, 'id')
    value = self.__get_prop (node, 'value')

    return id, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'part-model'
    self.name = 'Part Model'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Part/part-number model'
    self.mediator = mediator.copy ()
    self.icon_data = ICON_DATA

    # specific data
    self.__loaded = False
    self.__parts = {}
    self.__path = self.mediator.call ('app.get-path', 'part-catalogue.xml')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('part.new', self.svc_part_new)
    self.mediator.advertise ('part.set', self.svc_part_set)
    self.mediator.advertise ('part.get', self.svc_part_get)
    self.mediator.advertise ('part.remove', self.svc_part_remove)
    self.mediator.advertise ('part.catalogue.iter', self.svc_part_catalogue_iter)
    self.mediator.advertise ('part.catalogue.load', self.svc_part_catalogue_load)
    self.mediator.advertise ('part.catalogue.save', self.svc_part_catalogue_save)
    self.mediator.advertise ('part.catalogue.export', self.svc_part_catalogue_export)
    self.mediator.advertise ('part.catalogue.import', self.svc_part_catalogue_import)
    self.mediator.connect ('item.attribute-modified', self.on_item_attribute_modified)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load catalogue
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def load_catalogue (self, path):

    # @deprecated read catalogue version prior to Mobius 0.4.4, too
    data = open (path).read ()

    if '<catalogue>' in data:
      pickle = Pickle ()
      parts = {}
      for part in pickle.load (path):
        cat_dict = parts.setdefault (part.category, {})
        cat_dict[part.id] = part
    else:
      parts = self.mediator.call ('xml.unpickle', path)

    return parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load on demand
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def load_on_demand (self):
    if not self.__loaded:
      if os.path.exists (self.__path):
        self.__parts = self.load_catalogue (self.__path)
      else:
        self.__parts = {}
      self.__loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_new (self, category, part_id):
    self.load_on_demand ()

    part = Part ()
    part.category = category
    part.id = part_id
    self.svc_part_set (part)

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.set
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_set (self, part):
    self.load_on_demand ()

    cat_dict = self.__parts.setdefault (part.category, {})
    cat_dict[part.id] = part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_get (self, category, part_id):
    self.load_on_demand ()

    cat_dict = self.__parts.get (category, {})
    part = cat_dict.get (part_id)

    return part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_remove (self, part):
    self.load_on_demand ()

    cat_dict = self.__parts.get (part.category, {})
    old_part = cat_dict.pop (part.id, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.iter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_iter (self):
    self.load_on_demand ()

    for cat in self.__parts.itervalues ():
      for part in cat.itervalues ():
        yield part

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.load
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_load (self):
    self.__loaded = False
    self.load_on_demand ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_save (self):
    self.load_on_demand ()
    self.mediator.call ('xml.pickle', self.__path, self.__parts)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.export
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_export (self, path):
    self.load_on_demand ()
    self.mediator.call ('xml.pickle', path, self.__parts)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: part.catalogue.import
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_part_catalogue_import (self, path, method):
    self.load_on_demand ()

    # load catalogue
    parts = self.load_catalogue (path)

    # import
    if method in (METHOD_NEW_PART, METHOD_MERGE):
      for cat_dict in parts.itervalues ():
        for part in cat_dict.itervalues ():
          old_part = self.svc_part_get (part.category, part.id)

          if not old_part:
            self.svc_part_set (part)

          elif method == METHOD_MERGE:
            for attr_id, attr_value in part.attributes.iteritems ():
              if attr_value:
                old_part.attributes[attr_id] = attr_value

    elif method == METHOD_REPLACE:
      self.__parts = parts

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event handler: on_item_attribute_modified
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_item_attribute_modified (self, item, attr, text, new_text):
    if attr == 'part_id' and new_text != '':
      part = self.svc_part_get (item.category, new_text)

      if not part:
        self.mediator.emit ('part.unknown', item.category, new_text)

      else:    # fill item attributes
        for attr, value in part.attributes.iteritems ():
          self.mediator.call ('item.set-attribute', item, attr, value)
        self.mediator.call ('item.expand-initvalues', item)
