# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import mobius

(ITEM_ICON, ITEM_NAME, ITEM_OBJ) = range (3)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Case treeview
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class CaseTreeView (gtk.TreeView):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, *args):

    datastore = gtk.TreeStore (gtk.gdk.Pixbuf, str, object)
    gtk.TreeView.__init__ (self, datastore)
    self.set_headers_visible (False)
    self.set_enable_tree_lines (True)
    #self.connect ('button-press-event', self.on_button_press)
    self.connect ('cursor-changed', self.on_cursor_changed)
    self.connect ('test-expand-row', self.on_test_expand_row)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', ITEM_ICON)
    self.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', ITEM_NAME)
    self.append_column (tvcolumn)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_case (self, case):
    model = self.get_model ()
    model.clear ()

    # insert case and case children
    iter = self.add_object (case)
    self.case = case

    # expand case, if it has children
    if case.has_child ():
      path = model.get_string_from_iter (iter)
      self.expand_row (path, False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add an object to treeview, including its sub-objects
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_object (self, obj, iter=None):
    model = self.get_model ()

    # eliminate parent placeholder, if necessary
    if iter:
      child_iter = model.iter_children (iter)

      if child_iter:
        child_obj = model.get_value (child_iter, ITEM_OBJ)

        if not child_obj:
          model.remove (child_iter)

    # insert object
    icon = obj.get_icon ()
    name = obj.get_attribute ('name')
    obj_iter = model.append (iter, (icon, name, obj))

    # insert place holder if obj has sub-items
    if obj.has_child ():
      model.append (obj_iter)

    # expand item at iter
    if iter:
      path = model.get_string_from_iter (iter)
      self.expand_row (path, False)

    # return iterator
    return obj_iter

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add child to currently selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_child (self, obj):
    model, iter = self.get_selection ().get_selected ()

    if iter:
      parent = model.get_value (iter, ITEM_OBJ)
      parent.add_child (obj)
      self.add_object (obj, iter)
      mobius.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Get selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_selected_item (self):
    model, iter = self.get_selection ().get_selected ()

    if iter:
      item = model.get_value (iter, ITEM_OBJ)
    else:
      item = None
   
    return item

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Remove selected item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def remove_selected_item (self):
    model, iter = self.get_selection ().get_selected ()
    if iter:
      item = model.get_value (iter, ITEM_OBJ)
      model.remove (iter)

      self.case.set_modified (True)
      mobius.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle test_expand_row event
  # @detail This function implements lazy evaluation. Every item inserted
  # receive a null child (obj = None). When row is expanded, three situations
  # arise:
  # 1. row has no child: object already loaded but no subitems were found
  # 2. row has null child: just a markplace. Object must be loaded
  # 3. row has valid child: object already loaded
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_test_expand_row (self, treeview, iter, path, *args):
    model = treeview.get_model ()
    obj = model.get_value (iter, ITEM_OBJ)
    child_iter = model.iter_children (iter)

    if child_iter:
      child_obj = model.get_value (child_iter, ITEM_OBJ)

      if not child_obj:
        model.remove (child_iter)

        for i in obj:
          self.add_object (i, iter)

    return False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle cursor changed event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_cursor_changed (self, treeview, *args):
    model, iter = treeview.get_selection ().get_selected ()

    if iter:
      obj = model.get_value (iter, ITEM_OBJ)

      if model.get_path (iter) == (0,):
        mobius.mediator.emit ('case-selected', obj)
      else:
        mobius.mediator.emit ('item-selected', obj)
