<?php
/* LARUS BOARD ========================================================
 * Encoded in UTF-8 (micro symbol: µ)
 * Copyright © 2008,2009 by "The Larus Board Team"
 * This file is part of "Larus Board".
 *
 * "Larus Board" is free software: you can redistribute it and/or modify
 * it under the terms of the modified BSD license.
 *
 * "Larus Board" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * You should have received a copy of the modified BSD License
 * along with this package. If not, see
 * <http://download.savannah.gnu.org/releases/larusboard/COPYING.BSD>.
 */
  if ( !defined('__XF_INCLUDE') )
  die('File "'.basename(__FILE__).'" cannot be executed directly!');
  if ( !class_exists('XF') )
  die('Root class is not loaded, yet!');

/**
* XFImage abstracts php image function to draw bitmap and vector based
* @package lbbackend
*/
class XFImage {
/**
* @var integer filter flag: no filter
*/
const FILTER_NONE = 0x00;
/**
* @var integer filter flag: smooth edges used on filledrectangle() [vector only]
*/
const FILTER_SMOOTH_EDGE = 0x01;
/**
* @var integer filter flag: draw stronger lines used in line()
*/
const FILTER_LINE_STRONG = 0x02;
/**
* @var integer filter flag: draw a background gradient on filledrectangle() [vector only]
*/
const FILTER_GRADIENT = 0x04;
/**
* @var array queue of render jobs
*/
protected static $work_queue = array();
/**
* @var array resource handlers for bitmap images
*/
protected $handler_bitmap = array();
/**
* @var array resource handlers for vector images
*/
protected $handler_vector = array();
/**
* @var array meta data for images
*/
protected $handler_meta = array();
/**
* @var array resource handlers for colors in bitmap images
*/
protected $color_bitmap = array();
/**
* @var array resource handlers for colors in vector images
*/
protected $color_vector = array();
/**
* @var array handle filters
*/
protected $filter = array('id'=>0,'data'=>'');

  /**
  * enable the image processor
  * @return boolean
  * @since 1.0.0
  */
  public function enabled(){
    if ( extension_loaded('gd') ){
    $a = gd_info();
      if ( $a['PNG Support'] && function_exists('imagecreatetruecolor') )
      return true;
    }
  return false;
  }

  /**
  * create an image. returns the id for all later operations.
  * @param integer $size_x width in pixels
  * @param integer $size_y height in pixels
  * @param string $option enable smoother lines with 'antialias=yes'
  * @return integer
  * @since 1.0.0
  */
  public function create($size_x,$size_y,$option = ''){
  $id = intval(mt_rand(0,PHP_INT_MAX));
  self::queue('create:'.$id,array('size_x'=>(int)$size_x,'size_y'=>(int)$size_y,'option'=>$option));
  $this->handler_meta[$id] = array('width'=>(int)$size_x,'height'=>(int)$size_y);
  $this->color($id,0x00,0x00,0x00,1,'black');
  $this->color($id,0xff,0xff,0xff,1,'white');
  $this->color($id,0xff,0x00,0x00,1,'red');
  $this->color($id,0x00,0xff,0x00,1,'green');
  $this->color($id,0x00,0x00,0xff,1,'blue');
  return $id;
  }

  /**
  * set a color. channels can be in hexadecimal '0x00-0xff' or integer '0-255'
  * @param integer $id image id from XFImage::create()
  * @param integer $chr red channel
  * @param integer $chr green channel
  * @param integer $chb blue channel
  * @param double $cha alpha channel
  * @param string $name name of color for referencing
  * @return true
  * @since 1.0.0
  */
  public function color($id,$chr,$chg,$chb,$cha,$name){
  $name = preg_replace('/[^a-z0-9_]+/iu','',$name);
    if ( $chr < 0 || $chr > 255 )
    $chr = 128;
    if ( $chg < 0 || $chg > 255 )
    $chr = 128;
    if ( $chb < 0 || $chb > 255 )
    $chr = 128;
    if ( $cha < 0 || $cha > 1 )
    $cha = 1;
  return self::queue('color:'.(int)$id,array('name'=>$name,'red'=>(int)$chr,'green'=>(int)$chg,'blue'=>(int)$chb,'alpha'=>(double)$cha));
  }

  /**
  * set background color
  * @param integer $id image id from XFImage::create()
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function background($id,$color){
  $this->filledrectangle((int)$id,0,0,$this->handler_meta[(int)$id]['width'],$this->handler_meta[(int)$id]['height'],$color);
  return true;
  }

  /**
  * paint a border with one pixel at the edge
  * @param integer $id image id from XFImage::create()
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function border($id,$color){
  $this->rectangle((int)$id,0,0,$this->handler_meta[(int)$id]['width']-1,$this->handler_meta[(int)$id]['height']-1,$color);
  return true;
  }

  /**
  * paint a line
  * @param integer $id image id from XFImage::create()
  * @param integer $x1 left-top x coordinate
  * @param integer $y1 left-top y coordinate
  * @param integer $x2 right-bottom x coordinate
  * @param integer $y2 right-bottom y coordiate
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function line($id,$x1,$y1,$x2,$y2,$color){
  return self::queue('line:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'x2'=>(int)$x2,'y2'=>(int)$y2,'color'=>$color));
  }

  /**
  * paint an unfilled rectangle
  * @param integer $id image id from XFImage::create()
  * @param integer $x1 left-top x coordinate
  * @param integer $y1 left-top y coordinate
  * @param integer $x2 right-bottom x coordinate
  * @param integer $y2 right-bottom y coordinate
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function rectangle($id,$x1,$y1,$x2,$y2,$color){
  return self::queue('rectangle:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'x2'=>(int)$x2,'y2'=>(int)$y2,'color'=>$color));
  }

  /**
  * paint a filled rectangle
  * @param integer $id image id from XFImage::create()
  * @param integer $x1 left-top x coordinate
  * @param integer $y1 left-top y coordinate
  * @param integer $x2 right-bottom x coordinate
  * @param integer $y2 right-bottom y coordinate
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function filledrectangle($id,$x1,$y1,$x2,$y2,$color){
  return self::queue('filledrectangle:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'x2'=>(int)$x2,'y2'=>(int)$y2,'color'=>$color));
  }

  /**
  * paint an unfilled arc
  * @param integer $id image id from XFImage::create()
  * @param integer $x1 x coordinate
  * @param integer $y1 y coordinate
  * @param integer $w width in pixels
  * @param integer $h height in pixels
  * @param integer $start start angle
  * @param integer $end end angle
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function arc($id,$x1,$y1,$w,$h,$start,$end,$color){
  return self::queue('arc:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'w'=>(int)$w,'h'=>(int)$h,'color'=>$color));
  }

  /**
  * paint a filled arc
  * @param integer $id image id from XFImage::create()
  * @param integer $x1 x coordinate
  * @param integer $y1 y coordinate
  * @param integer $w width in pixels
  * @param integer $h height in pixels
  * @param integer $start start angle
  * @param integer $end end angle
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function filledarc($id,$x1,$y1,$w,$h,$start,$end,$color){
  return self::queue('filledarc:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'w'=>(int)$w,'h'=>(int)$h,'color'=>$color));
  }

  /**
  * paint a string
  * @param integer $id image id from XFImage::create()
  * @param integer $size letter size (1-5)
  * @param integer $x1 x coordinate
  * @param integer $y1 y coordinate
  * @param string $msg text message
  * @param string $color name of color
  * @return true
  * @since 1.0.0
  */
  public function string($id,$size,$x1,$y1,$msg,$color){
  return self::queue('text:'.(int)$id,array('x1'=>(int)$x1,'y1'=>(int)$y1,'size'=>(int)$size,'msg'=>$msg,'color'=>$color));
  }

  /**
  * apply a filter for next operations
  * @param integer $filter filter id
  * @param string $data options
  * @return true
  * @since 1.0.0
  */
  public function use_filter($filter,$data = ''){
  return self::queue('filter-enable',array('id'=>(int)$filter,'data'=>$data));
  }

  /**
  * remove a filter from using
  * @return true
  * @since 1.0.0
  */
  public function reset_filter(){
  return self::queue('filter-disable',array());
  }

  /**
  * output the rendered image
  * @param integer $id image id from XFCreate::image()
  * @param string $format use 'png' or 'svg'
  * @return resource
  * @since 1.0.0
  */
  public function output($id,$format){
    if ( headers_sent() )
    return false;
    switch ( strtolower($format) ){
    case 'png':
    $this->render('bitmap');
    header('Content-Type: image/png');
    imagepng($this->handler_bitmap[(int)$id]);
    break;
    case 'svg':
    $this->render('vector');
    $proc = $this->svg_skeleton();
    $proc = str_replace('{$SIZE_X}',$this->handler_meta[(int)$id]['width'].'px',$proc);
    $proc = str_replace('{$SIZE_Y}',$this->handler_meta[(int)$id]['height'].'px',$proc);
    $proc = str_replace('{$CONTENT}',implode(chr(10),$this->handler_vector[(int)$id]),$proc);
    header('Content-Type: image/svg+xml');
    echo $proc;
    break;
    }
  return true;
  }

  /**
  * add a job to work queue
  * @param string $a job name and image identifier, separated by :
  * @param array $b job arguments
  * @return true
  * @since 1.0.0
  */
  protected static function queue($a,$b){
  $a = explode(':',$a);
  self::$work_queue[] = array('function'=>$a[0],'imageid'=>(isset($a[1]))?(int)$a[1]:0,'args'=>$b);
  return true;
  }

  /**
  * execute jobs from work queue and render the image
  * @param string $mode render 'bitmap' image or 'vector' type, 'purge' cleans queue
  * @return true
  * @since 1.0.0
  */
  protected function render($mode){
    if ( $mode === 'purge' ){
    self::$work_queue = array();
    return true;
    }
    foreach ( self::$work_queue as $job ){
      switch ( $job['function'] ){
      case 'create':
      $job['args']['option'] = explode(',',$job['args']['option']);
        if ( $mode === 'bitmap' ){
        $this->handler_bitmap[$job['imageid']] = imagecreatetruecolor($job['args']['size_x'],$job['args']['size_y']);
          if ( in_array('antialias=yes',$job['args']['option'],true) && function_exists('imageantialias') )
          imageantialias($this->handler_bitmap[$job['imageid']],true);
        }
        elseif ( $mode === 'vector' )
        $this->handler_vector[$job['imageid']] = array();
      break;
      case 'color':
        if ( $mode === 'bitmap' ){
        $this->color_bitmap[$job['args']['name']] = ( $job['args']['alpha'] === 1 )
        ? imagecolorallocate($this->handler_bitmap[$job['imageid']],$job['args']['red'],$job['args']['green'],$job['args']['blue'])
        : imagecolorallocatealpha($this->handler_bitmap[$job['imageid']],$job['args']['red'],$job['args']['green'],$job['args']['blue'],0x7f-($job['args']['alpha']*0x7f));
        }
        elseif ( $mode === 'vector' ){
        $this->color_vector[$job['args']['name']] = '"'.sprintf('#%02x%02x%02x',$job['args']['red'],$job['args']['green'],$job['args']['blue']).'"';
          if ( $job['args']['alpha'] !== 1 )
          $this->color_vector[$job['args']['name']] .= ' fill-opacity="'.$job['args']['alpha'].'"';
        }
      break;
      case 'line':
        if ( $mode === 'bitmap' ){
          // NOTE: yes, i am aware of imagesetthickness() but it is not available (currently), if image is antialiased...
          if ( $this->filter['id']&self::FILTER_LINE_STRONG ){
          imageline($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1']-1,$job['args']['x2'],$job['args']['y2']-1,$this->color_bitmap[$job['args']['color']]);
          imageline($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1']+1,$job['args']['x2'],$job['args']['y2']+1,$this->color_bitmap[$job['args']['color']]);
          }
        imageline($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1'],$job['args']['x2'],$job['args']['y2'],$this->color_bitmap[$job['args']['color']]);
        }
        elseif ( $mode === 'vector' ){
        $sw = ( $this->filter['id']&self::FILTER_LINE_STRONG ) ? '3px' : '1px';
        $this->handler_vector[$job['imageid']][] = '<line x1="'.doubleval($job['args']['x1']+0.5).'" y1="'.doubleval($job['args']['y1']+0.5).'" x2="'.doubleval($job['args']['x2']+0.5).'" y2="'.doubleval($job['args']['y2']+0.5).'" stroke-width="'.$sw.'" stroke='.$this->color_vector[$job['args']['color']].' />';
        }
      break;
      case 'rectangle':
        if ( $mode === 'bitmap' )
        imagerectangle($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1'],$job['args']['x2'],$job['args']['y2'],$this->color_bitmap[$job['args']['color']]);
        elseif ( $mode === 'vector' )
        $this->handler_vector[$job['imageid']][] = '<rect x="'.doubleval($job['args']['x1']+0.5).'" y="'.doubleval($job['args']['y1']+0.5).'" width="'.doubleval($job['args']['x2']-$job['args']['x1']).'" height="'.doubleval($job['args']['y2']-$job['args']['y1']).'" fill="none" stroke-width="1px" stroke='.$this->color_vector[$job['args']['color']].' />';
      break;
      case 'filledrectangle':
        if ( $mode === 'bitmap' )
        imagefilledrectangle($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1'],$job['args']['x2'],$job['args']['y2'],$this->color_bitmap[$job['args']['color']]);
        elseif ( $mode === 'vector' ){
        $se = ( $this->filter['id']&self::FILTER_SMOOTH_EDGE ) ? 'rx="3"' : '';
          if ( $this->filter['id']&self::FILTER_GRADIENT ){
          $f = explode(',',$this->filter['data']);
          $this->handler_vector[$job['imageid']][] = '<linearGradient id="'.preg_replace('/[^a-z0-9_]+/iu','',$f[0]).'" gradientTransform="rotate(60)">
            <stop offset="'.intval($f[1]).'%" stop-color='.$this->color_vector[$f[3]].' />
            <stop offset="'.intval($f[2]).'%" stop-color='.$this->color_vector[$f[4]].' />
          </linearGradient>';
          $gr = 'fill="url(#'.preg_replace('/[^a-z0-9_]+/iu','',$f[0]).')"';
          }
          else
          $gr = 'fill='.$this->color_vector[$job['args']['color']];
        $this->handler_vector[$job['imageid']][] = '<rect x="'.$job['args']['x1'].'" y="'.$job['args']['y1'].'" '.$se.' width="'.doubleval($job['args']['x2']-$job['args']['x1']).'" height="'.doubleval($job['args']['y2']-$job['args']['y1']).'" '.$gr.' />';
        }
      break;
      case 'arc':
        if ( $mode === 'bitmap' )
        imagearc($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1'],$job['args']['w'],$job['args']['h'],0,360,$this->color_bitmap[$job['args']['color']]);
        elseif ( $mode === 'vector' )
        $this->handler_vector[$job['imageid']][] = '<ellipse cx="'.$job['args']['x1'].'" cy="'.$job['args']['y1'].'" rx="'.doubleval($job['args']['w']/2).'" ry="'.doubleval($job['args']['h']/2).'" fill="none" stroke-width="1px" stroke='.$this->color_vector[$job['args']['color']].' />';
      break;
      case 'filledarc':
        if ( $mode === 'bitmap' )
        imagefilledarc($this->handler_bitmap[$job['imageid']],$job['args']['x1'],$job['args']['y1'],$job['args']['w'],$job['args']['h'],0,360,$this->color_bitmap[$job['args']['color']]);
        elseif ( $mode === 'vector' )
        $this->handler_vector[$job['imageid']][] = '<ellipse cx="'.$job['args']['x1'].'" cy="'.$job['args']['y1'].'" rx="'.doubleval($job['args']['w']/2).'" ry="'.doubleval($job['args']['h']/2).'" fill='.$this->color_vector[$job['args']['color']].' />';
      break;
      case 'text':
        // NOTE: imagestring() cannot handle unicode properly, so we decode it...
        if ( $mode === 'bitmap' )
        imagestring($this->handler_bitmap[$job['imageid']],$job['args']['size'],$job['args']['x1'],$job['args']['y1'],utf8_decode($job['args']['msg']),$this->color_bitmap[$job['args']['color']]);
        elseif ( $mode === 'vector' ){
        $sizemap = array(1=>8,2=>10,3=>12,4=>14,5=>15);
        $this->handler_vector[$job['imageid']][] = '<text x="'.$job['args']['x1'].'" y="'.doubleval($job['args']['y1']+$sizemap[$job['args']['size']]).'" font-family="courier" font-size="'.$sizemap[$job['args']['size']].'px" fill='.$this->color_vector[$job['args']['color']].'>'.htmlspecialchars($job['args']['msg']).'</text>';
        }
      break;
      case 'filter-enable':
      $this->filter['id'] = $job['args']['id'];
      $this->filter['data'] = $job['args']['data'];
      break;
      case 'filter-disable':
      $this->filter['id'] = self::FILTER_NONE;
      $this->filter['data'] = '';
      break;
      case 'destroy':
        if ( $mode === 'bitmap' ){
        imagedestroy($this->handler_bitmap[$job['imageid']]);
        $this->color_bitmap[$job['imageid']] = array();
        unset($this->handler_meta[$job['imageid']]);
        }
        elseif ( $mode === 'vector' ){
        unset($this->handler_vector[$job['imageid']]);
        $this->color_vector[$job['imageid']] = array();
        unset($this->handler_meta[$job['imageid']]);
        }
      break;
      }
    }
  return true;
  }

  /**
  * frame for svg files
  * @return string
  * @since 1.0.0
  */
  protected function svg_skeleton(){
  return '<?xml version="1.0" encoding="utf-8"?>
  <!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN"  "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
  <svg xmlns="http://www.w3.org/2000/svg"
     xmlns:xlink="http://www.w3.org/1999/xlink" xmlns:ev="http://www.w3.org/2001/xml-events"
     version="1.1" baseProfile="tiny"
     width="{$SIZE_X}" height="{$SIZE_Y}">
  <g id="larus_boars_statistic_image">
  {$CONTENT}
  </g>
  </svg>';
  }

}
?>