/*
 	This file is part of jrisk.

    jrisk is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    jrisk is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with jrisk; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 14-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

/**
 *
 * Gestisce gli scontri fra due giocatori. Calcola le armate di
 * ognuno sui territori coinvolti, gestisce il lancio dei dadi e
 * stabilisce l'eventuale vincitore. Dunque, aggiorna lo stato del
 * gioco. Incorpora le regole dei combattimenti della versione
 * standard di Risiko.
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class Battaglia {
    /** Numero massimo di dadi per lancio */
    private final int MAX_DADI = 3;
    /** Attaccante */
    private Giocatore attacco;
    /** Difensore */
    private Giocatore difesa;
    /** <code>Territorio</code> dal quale parte l'attacco */
    private Territorio da;
    /** <code>Territorio</code> della difesa */
    private Territorio a;
    /** Tiro di dadi dell'attaccante */
    private SerieDiDadi lancioAttacco;
    /** Tiro di dadi del difensore */
    private SerieDiDadi lancioDifesa;
    
    /**
     * Controlla che il numero di dadi da lanciare per l'attacco e/o
     * per la difesa sia accettabile, tenendo conto del numero massimo
     * di dadi stabilito dalle regole (MAX_DADI) e dal numero di armate
     * presenti sul territorio.
     * 
     * @param n dadi da lanciare
     * @param t territorio attaccante/difensore
     * @return il numero di dadi possibili da lanciare
     */
    private int impostaDadi(int n, Territorio t) {
        if(n > MAX_DADI)
            n = MAX_DADI;
        if(n > t.getArmate())
            n = t.getArmate();
        return n;
    }
    /**
     * Inizializza un oggetto gestore degli scontri fra i territori di
     * giocatori diversi.
     * 
     * @param attaccante giocatore in attacco
     * @param difensore giocatore in difesa
     * @param territorioAttaccante territorio dell'attaccante
     * @param territorioDifensore territorio del difensore
     */
    public Battaglia(Giocatore attaccante,
            Giocatore difensore,
            String territorioAttaccante,
            String territorioDifensore) {
        this.attacco = attaccante;
        this.difesa = difensore;
        this.da = this.attacco.getTerritorio(territorioAttaccante);
        this.a = this.difesa.getTerritorio(territorioDifensore);        
    }
    
    /**
     * <p>Uno scontro fra due territori di giocatori diversi prevede il lancio
     * di dadi da parte dell'attacco e dunque un lancio di dadi da parte della difesa.
     * Viene confrontato un numero comune di dadi fra l'attacco e la difesa. Un dado
     * che sia maggiore (o uguale, nel caso del difensore) del dado corrispondente
     * dell'avversario permette al giocatore di eliminare un'armata dal territorio
     * dell'avversario. Nel caso che uno dei due territori rimanga sprovvisto di armate
     * allora esso passa al giocatore che possiede l'altro territorio.</p>
     * 
     * @param nDadiAttacco numero di dadi lanciato dall'attaccante
     * @param nDadiDifesa numero di dadi lanciati dal difensore
     * @return un valore diverso da 0 se la battaglia ha comportato la perdita del
     * territorio da parte di uno dei due giocatori, maggiore di 0 in caso di vittoria
     * dell'attaccante (il valore indica quante armate sono state impegnate nell'attacco),
     * minore di 0 in caso di vittoria del difensore.
     */
    public int scontro(int nDadiAttacco, int nDadiDifesa) {
        //SerieDiDadi lancioAttacco;						// Dadi lanciati dall'attaccante
        //SerieDiDadi lancioDifesa;						// Dadi lanciati dal difensore
        int nDadi;										// Numero di dati da confrontare
        int esito;										// Esito del lancio dei dadi
        ComparatoreSerieDiDadi comparatore;				// Comparatore di lanci
        
        /* Lancia i dadi dell'attaccante e del difensore */
        lancioAttacco = new SerieDiDadi(impostaDadi(nDadiAttacco, da));
        lancioDifesa = new SerieDiDadi(impostaDadi(nDadiDifesa, a));
        
        comparatore = new ComparatoreSerieDiDadi();		// Inizializzo l'oggetto comparatore
        /* Numero dadi confrontati */
        nDadi = comparatore.getDadiDaConfrontare(lancioAttacco.getNumeroDadi(),lancioDifesa.getNumeroDadi());
        /* Esito della "battaglia" */
        esito = comparatore.compare(lancioAttacco, lancioDifesa);

        /* Interpeta l'esito del lancio dei dadi
         * e ne calcola le conseguenze
         */
        if(esito > 0) {									// Se ha vinto l'attacco
            da.setArmate(da.getArmate() - (nDadi - esito)); // Elimina le armate perse dall'attacco
            a.setArmate(a.getArmate() - esito);			// elimina le armate perse dalla difesa
            if(a.isVuoto()) {							// Se il difensore ha perso il territorio
                attacco.setPuoPescare(true);			// al prossimo turno l'attaccante potr pescare un carta
                return lancioAttacco.getNumeroDadi();	// restituisci il numero di armate usate dall'attacco
            }
                                      
        } else											// Se ha vinto la difesa
            da.setArmate(da.getArmate() - nDadi);		// elimina le armate perse dal territorio dell'attaccante
        
    	if(da.isVuoto()) {								// Se l'attaccante ha perso il territorio
    	    difesa.setPuoPescare(true);					// al prossimo turno il difensore potr pescare una carta
    	    return -1;									// restituisce un valore negativo
    	}
	    /* se lo scontro nn ha comportato perdite di territori
    	 * da parte ne del difensore ne dell'attaccante, restituisce
    	 * 0
    	 */
    	return 0;
    }
    
    /**
     * Restituisce il lancio di dadi dell'attacco.
     * 
     * @return un riferimento ad un oggetto <code>SerieDiDadi</code>,
     * con il risultato dei dadi lanciati dall'attaccante
     */
    public SerieDiDadi getLancioAttacco() { return this.lancioAttacco; }
    /**
     * Restituisce il lancio di dadi della difesa.
     * 
     * @return un riferimento ad un oggetto <code>SerieDiDadi</code>,
     * con il risultato dei dadi lanciati dal difensore
     */
    public SerieDiDadi getLancioDifesa() { return this.lancioDifesa; }
}
