/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Copyright  2005 Alessio Treglia 
 */

import java.util.Random;
/**
 * <p>Gestisce un dado e ne simula il comportamento.
 * Il lancio di un dado tradizionale a 6 facce comporta
 * l'uscita di un numero compreso fra 1 e 6 (inclusi).
 * Usa l'algoritmo di generazione di numeri pseudo-casuali
 * implementato dalla classe Random inclusa nel pacchetto
 * java.util della libreria. Gli oggetti di questo tipo
 * sanno confrontarsi fra loro, grazie al metodo compareTo()
 * implementato dall'interfaccia Comparable.</p>
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 * @see java.util.Random
 */
public class Dado implements Comparable {
    /** Numero di facce del dado */
    private final int FACCE_DEL_DADO = 6;
    /** Contiene l'ultimo numero uscito */
    private int ultimoLancio;
    /** Generatore di numeri casuale */
    private Random rand;
    /**
     * Costruttore no-arg. Inizializza il generatore di numeri casuali.
     */
    public Dado() {
        rand = new Random();
        ultimoLancio = lancio();
    }
    
    /**
     * Simula il lancio di un dado a 6 facce.
     * Memorizza il numero uscito nell'attributo privato
     * ultimoLancio.
     * 
     * @return un numero (pseudo-casuale) compreso
     * fra 1 e 6 (inclusi).
     */
    public int lancio() {
        ultimoLancio = rand.nextInt(FACCE_DEL_DADO) + 1;
        return ultimoLancio;
    }
    /**
     * @return il numero uscito nell'ultimo lancio
     */
    public int getUltimoLancio() {
        return ultimoLancio;
    }
    /**
     * Confronta il valore dell'ultimo lancio di ogni dado.
     * 
     * @param ob <code>Dado</code> da confrontare
     * @return un valore minore, uguale o maggiore di 0 a seconda
     * che l'oggetto chiamante sia minore, uguale o maggiore dell'
     * oggetto del parametro.
     */
    public int compareTo(Object ob) {
        Dado altroDado = (Dado) ob;
        if(this.ultimoLancio < altroDado.getUltimoLancio())
            return -1;
        if(this.ultimoLancio > altroDado.getUltimoLancio())
            return 1;
        return 0;
    }
    
    /**
     * @return una descrizione stringa del valore estratto dal dado
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("[ " + Integer.toString(ultimoLancio) + " ]");
        return buffer.toString();
    }
}
