#! /usr/local/bin/scm \ %0 %*
- !#
;;; "nk" create, manage, and query spectra database
;;; Copyright (C) 2003, 2006, 2008 Aubrey Jaffer

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

(require 'filename)
(require 'scanf)
(require 'printf)
(require 'line-i/o)
(require 'string-case)
(require 'string-search)
(require 'metric-units)
(require 'rwb-isam)
(require 'databases)
(require 'database-interpolate)
(require 'eps-graph)
(require-if 'compiling 'read-command)

(if (not (defined? real-log10))
    (define real-log10 $log10))

(define (nk.usage)
  (display "\

Use the nk program to create, manage, and query a refractive-index
spectra database.

The nk program can read Sopra format files.  These files have a \".nk\"
suffix.  A bundle of spectra for over 275 materials is available from:
http://www.sopra-sa.com/fichiers/Database.zip (Nov 28, 2000).  Some
metal spectra included with FreeSnell also have a \".nk\" suffix, but
with a different format.

The zip file http://www.sspectra.com/files/misc/win/SOPRA.EXE
contains \"README.TXT\" identifying the Sopra files.

"
	   (current-error-port))
  (nki.usage))

(define (nki.usage)
  (display "\

The optional argument [-F path] sets the path to the database file to
be accessed or created.  If it is not given, then the environment
variable \"NK_DATABASE_PATH\" names the database file if defined,
defaulting to \"nk.rwb\" in the current directory.  The name for each
spectrum in the database is taken from its filename sans the \".nk\"
or \"ir.nk\" suffix.

\
Usage: nk [-F path]
\
  Starts the nk shell.  Type nk commands without the nk; ^Z to exit.
\
Usage: nk [-F path] --add FILE.nk ...
\
  Add spectra named FILE to database.
\
Usage: nk [-F path] --new FILE.nk ...
\
  Add new or replace spectra named FILE to database.
\
Usage: nk [-F path] --del 'GLOB' ...
\
  Delete spectra of names matching GLOB from database.
\
Usage: nk [-F path] --plot 'GLOB' ...
Usage: nk [-F path] --lin-lin 'GLOB' ...
Usage: nk [-F path] --log-log 'GLOB' ...
Usage: nk [-F path] --log-lin 'GLOB' ...
Usage: nk [-F path] --lin-log 'GLOB' ...
\
  Create encapsulated-PostScript (and view with Ghostview) spectra plots
  from database of names matching GLOB.
\
Usage: nk [-F path] --list 'GLOB' ...
\
  List names of spectra matching GLOB in database.
\
Usage: nk [-F path] NAME NUMBER.UNIT ...
\
  Prints NAME's n+k*i values at each NUMBER.UNIT from database.  The
  NUMBER can be fixed or floating point; the UNIT either \"eV\"
  (electron-volt) or \"m\" (meter) with an optional metric prefix or
  \"cm^-1\" (wave-number).
\

"
	   (current-error-port))
  (force-output (current-error-port))
  #f)

(define (nk.script args)
  (cond ((not (<= 1 (length args)))
	 (nki))
	((equal? "-F" (car args))
	 (cond ((null? (cdr args)) (nk.usage))
	       (else (set! *nkdb-filename* (cadr args))
		     (nk.script (cddr args)))))
	((eqv? #\- (string-ref (car args) 0))
	 (case (string->symbol (strip-leading-dashes (car args)))
	   ((log-log) (apply plot-nk-spectra #t #t (cdr args)))
	   ((lin-log) (apply plot-nk-spectra #t #f (cdr args)))
	   ((log-lin) (apply plot-nk-spectra #f #t (cdr args)))
	   ((lin-lin
	     plot)    (apply plot-nk-spectra #f #f (cdr args)))
	   ((list) (apply list-nk-spectra (cdr args)))
	   ((new) (apply add-nk-spectra #t (cdr args)))
	   ((add) (apply add-nk-spectra #f (cdr args)))
	   ((del) (apply del-nk-spectra (cdr args)))
	   (else (nk.usage))))
	(else (apply nk-values args))))

(define (nki.prompt-read-command)
  (display "nk ")
  (force-output)
  (read-command))

(define (nki.welcome)
  (display
   (case (software-type)
     ((ms-dos windows atarist os/2)
      "Complete the nk command lines; ? for help; ^Z to quit.")
     (else
      "Complete the nk command lines; ? for help; ^D to quit.")))
  (newline))

(define (nki)				; interactive
  (require 'read-command)
  (nki.welcome)
  (do ((args (nki.prompt-read-command) (nki.prompt-read-command)))
      ((or (eof-object? args))
       (exit #t))
    (cond ((null? args))
	  ((equal? "?" (car args))
	   (nk.usage)
	   (nki.welcome))
	  (else
	   (nk.script args)))))

(define (strip-leading-dashes str)
  (define len (string-length str))
  (do ((idx 0 (+ 1 idx)))
      ((or (>= idx len)
	   (not (eqv? #\- (string-ref str idx))))
       (substring str idx len))))
				 ; Energy	     E =         h*c/L
(define eV<->L			 ; Photon wavelength L =         h*c/E
  (let ((c 299792458)		 ; speed of light    c =           m/s
	(h 6.62606876e-34)	 ; Plank's constant  h =           J.s
	(J/eV 1.602176462e-19))
    (lambda (eV) (/ (* h c) eV J/eV))))

(define (recip-cm x) (/ .01 x))

(define (readit port)
  (let ((num (read port)))
    (cond ((eqv? #\, (peek-char port))
	   (read-char port)))
    num))

(define (string-whitespace? str)
  (do ((idx (+ -1 (string-length str)) (+ -1 idx)))
      ((or (negative? idx)
	   (not (char-whitespace? (string-ref str idx))))
       (negative? idx))))

;;This is the filename of the nk database.
(define *nkdb-filename* (or (getenv "NK_DATABASE_PATH") "nk.rwb"))

;;Database is global.
(define *nkdb* #f)

;;Error routine.
(define (dno name) (slib:warn name 'database 'not 'opened *nkdb-filename*) #f)

(define (create-nk-database)
  (set! *nkdb* (create-database *nkdb-filename* 'rwb-isam))
  (cond (*nkdb*
	 (define-domains *nkdb*
	   '(wavelength #f real? r32 "m")
	   '(refractive-index #f complex? c32 ""))
	 (define-tables *nkdb*
	   '(*desc:spectrum*
	     *columns*
	     *columns*
	     ((1 #t wavelength #f wavelength)
	      (2 #f refractive-index #f refractive-index))))
	 (close-database *nkdb*)))
  (open-database! *nkdb-filename* 'rwb-isam))

(define (nk:open-database! filename type create?)
  (cond ((file-exists? filename)
	 (let ((nkdb (open-database! filename type)))
	   (cond ((not nkdb)
		  (describe-file-lock *nkdb-filename*)
		  (slib:warn 'could 'not 'open *nkdb-filename* 'for 'writing)))
	   nkdb))
	(create?
	 (let ((nkdb (create-nk-database)))
	   (cond ((not nkdb)
		  (slib:warn 'could 'not 'create *nkdb-filename*)))
	   nkdb))
	(else
	 (slib:warn 'could 'not 'find *nkdb-filename*)
	 #f)))

;;http://www.sopra-sa.com/more/database.asp
;;
;; Database n&k
;;
;; SOPRA provides with their instruments one of the largest database of
;; optical indices available in the world (278 materials ).  We have now
;; decided to give this database for free to the scientific community.
;; It can be download very easily here.  Nevertheless the inverse can
;; also be true and if you send us your own optical indices with the
;; allowance to add them in the database their can be useful to other
;; people.  This database comes with a help file which provide some more
;; details about these data and also their origin.
;;
;; File Format:
;;
;; The different NK files are in ASCII format (*.NK files).  The first
;; number indicates the spectral unit (1 = eV, 2 = m, 3 = cm-1, 4 = nm).
;; The second and the third ones indicate the spectral range and the
;; fourth one the number of intervals.  The N and K values are then
;; stored in the order N,K,N,K... One example is reported below:
;;
;; 1, 0.6, 5.9, 53
;; 3.4471, 0.
;; 3.4595, 0.
;; ...
;; 1.1328, 3.0447
;; 1.0831, 2.9823
;;
;;     * NK database with 278 materials.
;;       http://www.sopra-sa.com/fichiers/Database.zip

(define (load-spectrum.nk! path spectrum-table te?)
  (define (micronize wl) (* 1e-6 wl))
  (define (nanoize wl) (* 1e-9 wl))
  (define spectrum:insert (spectrum-table (if te? 'row:update 'row:insert)))
  (call-with-input-file path
    (lambda (port)
      (define first (readit port))
      (define (reader converter)
	(printf "%s: %s%s\\n" path first (read-line port)) ;flush rest of line
	(do ((line (read-line port) (read-line port)))
	    ((eof-object? line))
	  (let ((row (scanf-read-list " %g %g %g" line)))
	    (if (negative? (cadr row)) (slib:warn (car row) first 'negative 'k (cadr row)))
	    (if (negative? (caddr row)) (slib:warn (car row) first 'negative 'n (caddr row)))
	    (spectrum:insert
	     (list (converter (car row))
		   (make-rectangular (cadr row) (max 0 (caddr row))))))))
      (case first
	((eV) (reader eV<->L))
	((/cm) (reader recip-cm))
	((um) (reader micronize))
	((nm) (reader nanoize))
	(else
	 (let ((format (case first ((1) 'eV) ((2) 'um) ((3) '/cm) ((4) nm)
			     (else (slib:warn 'bad 'first 'line? first)
				   #f))))
	   (and
	    format
	    (let* ((conv-lm (case format
			      ((eV) identity)
			      ((um) micronize)
			      ((/cm) identity)
			      ((nm) nanoize)))
		   (conv-dt (case format
			      ((eV) eV<->L)
			      ((um) identity)
			      ((/cm) recip-cm)
			      ((nm) identity)))
		   (start   (readit port))
		   (end     (readit port))
		   (count-1 (readit port)))
	      (define wlinc (conv-lm (/ (- end start) count-1)))
	      (define cnt 0)
	      (define eline (read-line port))
	      (printf "%s: %.3g.%s :: %.3g.%s\\n" path start format end format)
	      (if (not (string-whitespace? eline)) (write-line eline))
	      (do ((wl (conv-lm start) (+ wl wlinc))
		   (val (readit port) (readit port)))
		  ((or (not (number? val))
		       (eqv? #\newline (peek-char port)))
		   (if (not (eqv? cnt (+ 1 count-1)))
		       (slib:warn 'Inserted cnt 'of (+ 1 count-1) 'values)))
		(let ((imag (readit port)))
		  (read-line port)
		  (if (negative? imag) (slib:warn 'negative 'k imag))
		  (if (negative? val) (slib:warn 'negative 'n val))
		  (cond ((not (number? imag)) (slib:warn 'strange 'imag imag))
			(else (spectrum:insert
			       (list (conv-dt wl)
				     (make-rectangular val (max 0 imag))))
			      (set! cnt (+ 1 cnt))))))))))))))

;;@body
;;@3 must be an open relational database, @2 a symbol, and the string
;;@1 must name an existing file with spectral values.  @0 creates a
;;table @2 in @3 and enters the associations found in @1 into it.
(define (file->spectrum! file table-name *nkdb* te?)
  (cond ((not (file-exists? file))	; logic error
	 (slib:error 'load-spectrum! 'file-exists? file)))
  (if (not te?)
      (define-tables *nkdb*
	`(,table-name
	  *desc:spectrum*
	  *desc:spectrum*
	  ())
	`(catalog
	  ((name symbol))
	  ()
	  ((,table-name)))))
  (let ((table ((*nkdb* 'open-table) table-name #t)))
    (cond (table (load-spectrum.nk! file table te?)))))

;;; Sopra/sicrir.nk and Sopra/sio2ir.nk extend the range of
;;; sicr.nk (crystalline Si) and sio2.nk respectively.
(define (add-nk-spectra new? . files)
  (define table-names
    (map (lambda (path)
	   (define len (string-length path))
	   (cond ((file-exists? path)
		  (let ((nam (substring path
					(+ 1 (or (string-reverse-index path #\/) -1))
					(+ -3 len))))
		    (set! len (string-length nam))
		    (string-ci->symbol
		     (if (and (> len 2)
			      (string-ci=? "ir" (substring nam (+ -2 len) len)))
			 (print nam '==> (substring nam 0 (+ -2 len)))
			 nam))))
		 (else (slib:warn 'add-nk-spectra 'can 'not 'find path)
		       (set! files #f)
		       path)))
	 files))
  (cond ((not files) (slib:warn 'add-nk-spectra 'not 'done) #f)
	(else
	 (set! *nkdb* (nk:open-database! *nkdb-filename* 'rwb-isam #t))
	 (and *nkdb*
	      (let ((table-exists? (*nkdb* 'table-exists?)))
		(cond (*nkdb*
		       (for-each (lambda (path name)
				   (define te? (table-exists? name))
				   (if (and new? te?)
				       ((*nkdb* 'delete-table) name))
				   (file->spectrum! path name *nkdb*
						    (and te? (not new?))))
				 files
				 table-names)
		       (close-database *nkdb*))
		      (else (dno 'add-nk-spectra))))))))

(define (find-matching *nkdb* globs)
  (define cat (open-table *nkdb* 'catalog))
  (define names '())
  (let ((catget* (and cat (cat 'get* 'name))))
    (and catget*
	 (if (null? globs)
	     (set! names (catget*))
	     (for-each
	      (lambda (glob)
		(define match? (filename:match-ci?? glob))
		(let ((finds (catget* (lambda (key)
					(match? (symbol->string key))))))
		  (cond ((null? finds) (slib:warn glob 'not 'found))
			(else (set! names (append finds names))))))
	      globs))
	 names)))

(define (del-nk-spectra . substances)
  (set! *nkdb* (nk:open-database! *nkdb-filename* 'rwb-isam #f))
  (and *nkdb*
       (let ((delete-table (*nkdb* 'delete-table))
	     (cat (open-table! *nkdb* 'catalog)))
	 (define catrem (and cat (cat 'row:remove)))
	 (for-each (lambda (substance)
		     (printf "removing %s\\n" substance)
		     (delete-table substance)
		     (catrem substance))
		   (find-matching *nkdb* substances)))
       (close-database *nkdb*)))

(define (array<-complex-table table log-y? log-x?)
  (define rows '())
  (define warp-y (if log-y? real-log10 identity))
  (define warp-x (if log-x? real-log10 identity))
  ((table 'for-each-row)
   (lambda (row)
     (set! rows (cons (list (warp-y (imag-part (cadr row)))
			    (warp-y (real-part (cadr row)))
			    (warp-x (car row)))
		      rows))))
  (let ((nra (make-array (A:floR64b)
			 (length rows)
			 (length (car rows)))))
    (do ((lns rows (cdr lns))
	 (idx (+ -1 (length rows)) (+ -1 idx)))
	((null? lns) nra)
      (do ((kdx (+ -1 (length (car rows))) (+ -1 kdx))
	   (lst (car lns) (cdr lst)))
	  ((null? lst))
	(array-set! nra (car lst) idx kdx)))))
;;(/ (imag-part (cadr row)) (real-part (cadr row)))

(define (graph-spectrum log-y? log-x? table-name table)
  (define tabra (array<-complex-table table log-y? log-x?))
  (create-postscript-graph
   (sprintf #f "/tmp/%s.eps" table-name) '(600 400)
   (whole-page)
   (setup-plot (column-range tabra 0)
	       (combine-ranges (column-range tabra 1) (column-range tabra 2)))
   (title-top table-name)
   (outline-rect plotrect)
   (in-graphic-context (set-linewidth 0) (grid-verticals) (grid-horizontals))
   (rule-vertical leftedge
		  (if log-y? "log refractive-index n" "refractive-index n")
		  10)
   (rule-horizontal bottomedge
		    (if log-x? "log(wavelength/m)" "wavelength in m")
		    10)
   (plot-column tabra 0 1 'line)
   (plot-column tabra 0 1 'plus)
   (set-color 'seagreen)
   (rule-vertical rightedge
		  (if log-y? "log extinction-coefficient k" "extinction-coefficient k")
		  -10)
   (plot-column tabra 0 2 'line)
   (plot-column tabra 0 2 'cross))
  (zero? (system (case (software-type)
		   ((ms-dos windows atarist os/2)
		    (sprintf #f
			     "%s \"\\\\tmp\\\\%s.eps\""
			     "\\Progra~1\\Ghostgum\\gsview\\gsview32"
			     table-name))
		   (else
		    (sprintf #f
			     "%s \"/tmp/%s.eps\""
			     "gv"
			     table-name))))))

(define (plot-nk-spectra log-y? log-x? . substances)
  (set! *nkdb* (open-database *nkdb-filename* 'rwb-isam))
  (cond (*nkdb*
	 (for-each
	  (lambda (name)
	    (graph-spectrum log-y? log-x? name (open-table *nkdb* name)))
	  (find-matching *nkdb* substances))
	 (close-database *nkdb*))
	(else (dno 'plot-nk-spectra))))

(define (nk-lookup substance itp wv)
  (define (errout) (printf "number? %s\\n" wv))
  (define (try un)
    (cond ((or (not (list? un)) (not (eqv? 2 (length un)))) #f)
	  ((positive? (si:conversion-factor "cm^-1" (cadr un)))
	   (let ((/cm (* (car un)
			 (si:conversion-factor "cm^-1" (cadr un)))))
	     (define w (recip-cm /cm))
	     (printf "%s: %.4g @ (%.3Km) (%.3g.eV) %g.cm^-1\\n"
		     substance (itp w) w (eV<->L w) /cm)))
	  ((positive? (si:conversion-factor "eV" (cadr un)))
	   (let ((eV (* (car un)
			(si:conversion-factor "eV" (cadr un)))))
	     (define w (eV<->L eV))
	     (printf "%s: %.4g @ (%.3Km) %.3g.eV (%g.cm^-1)\\n"
		     substance (itp w) w eV (recip-cm w))))
	  ((positive? (si:conversion-factor "m" (cadr un)))
	   (let ((w (* (car un)
		       (si:conversion-factor "m" (cadr un)))))
	     (define eV (eV<->L w))
	     (printf "%s: %.4g @ %.3Km (%.3g.eV) (%g.cm^-1)\\n"
		     substance (itp w) w eV (recip-cm w))))
	  (else #f)))
  (or (try (scanf-read-list "%g.%s" wv))
      (try (scanf-read-list "%d.%s" wv))
      (try (scanf-read-list "%g%s" wv))))

(define (nk-values substance . wvs)
  (set! *nkdb* (open-database *nkdb-filename* 'rwb-isam))
  (if (string? substance) (set! substance (string-ci->symbol substance)))
  (cond (*nkdb*
	 (let ((table (open-table *nkdb* substance)))
	   (and table
		(let ((itp (interpolate-from-table table 2)))
		  (for-each (lambda (wv)
			      (nk-lookup substance itp wv))
			    wvs))))
	 (close-database *nkdb*))
	(else (dno 'nk-values))))

(define (list-nk-spectra . globs)
  (define col 0)
  (define res 0)
  (set! *nkdb* (open-database *nkdb-filename* 'rwb-isam))
  (cond (*nkdb*
	 (for-each (lambda (name)
		     (define len (string-length (symbol->string name)))
		     (cond ((>= (+ col len res) 72)
			    (newline) (display name)
			    (set! col (+ len)))
			   (else
			    (display (make-string res #\space)) (display name)
			    (set! col (+ col len res))))
		     (set! res (modulo (- len) 12)))
		   (find-matching *nkdb* globs))
	 (newline)
	 (close-database *nkdb*))
	(else (dno 'list-nk-spectra))))

;;; procedure for "nk.bat".
(define (nk) (exit (nk.script (list-tail *argv* *optind*))))

;;(trace-all "/Program Files/slib/transact.scm")

(if *script* (nk))
