/*
 * BMovieReviewer Copyright (C) 2009 Michael J. Beer
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package data;

import java.io.File;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.util.Iterator;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.XMLEvent;

import data.Wrappers.Link;

public class XML {

	public static final String ZITAT = "zitat";

	public static final String ZITATE = "zitate";

	public static final String LINK = "link";

	public static final String LINKS = "links";
	
	public static final String LINKS_TYP = "typ";

	// public static final String PUNKT = "punkt";
	public static final String PUNKTWERTUNGEN = "punktwertungen";

	public static final String TEXTFELDER = "textfelder";

	public static final String BMOVIE = "bmovie";

	public static final String PUNKT_VAL = "val";

	public static final String PUNKT_ANMERKUNG = "note";
	
	public static final String COVER = "cover";

	protected XMLEventReader reader;

	protected Bogen bogen;

	public XML(Bogen bogen) {
		if (bogen == null) {
			throw new IllegalArgumentException();
		}
		this.bogen = bogen;
		System.out.println(bogen);
	}

	public XML(XMLEventReader reader, Bogen bogen) {
		if (reader == null) {
			throw new IllegalArgumentException();
		}
		if (bogen == null) {
			this.bogen = new Bogen();
		} else {
			this.bogen = bogen;
		}
		this.reader = reader;
	}

	public void printOpeningTag(PrintStream out, String tag) {
		out.println("<" + tag + ">");
	}

	public void printClosingTag(PrintStream out, String tag) {
		out.println("</" + tag + ">");
	}

	public void printHeader(PrintStream out) {
		out.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
	}

	public void printPunktWertung(PrintStream out, int i) {
		if (i >= Bogen.kategorien.length) {
			throw new IllegalArgumentException();
		}
		printOpeningTag(out, Bogen.kategorien[i] + " " + PUNKT_VAL + "=\""
				+ bogen.getPunkt(i) + "\" " + PUNKT_ANMERKUNG + "=\""
				+ bogen.getAnmerkung(i) + "\"");
		printClosingTag(out, Bogen.kategorien[i]);

	}

	public void printTextFeld(PrintStream out, int index) {
		printOpeningTag(out, Bogen.TEXTFELDER[index]);
		out.println(bogen.getText(index));
		printClosingTag(out, Bogen.TEXTFELDER[index]);
	}

	public void printZitat(PrintStream out, String zitat) {
		printOpeningTag(out, ZITAT);
		out.println(zitat.replaceAll("\"", "\\\""));
		printClosingTag(out, ZITAT);
	}

	public void printLink(PrintStream out, Link link) {
		printOpeningTag(out, LINK + " " + LINKS_TYP + "=\"" + Bogen.LINK_TYPES[link.getTyp()]
				+ "\"");
		out.println(link.getLink());
		printClosingTag(out, LINK);
		System.out.println("Link " + link.toString());
	}

	public void printCover(PrintStream out, String cover) {
		printOpeningTag(out, COVER);
		out.println(cover);
		printClosingTag(out, COVER);
	}
	
	public void printPunktWertungen(PrintStream out) {
		printOpeningTag(out, PUNKTWERTUNGEN);
		for (int index = 0; index < Bogen.kategorien.length; index++) {
			printPunktWertung(out, index);
		}
		printClosingTag(out, PUNKTWERTUNGEN);
	}

	public void printTextfelder(PrintStream out) {
		printOpeningTag(out, TEXTFELDER);
		for (int index = 0; index < Bogen.I_MAX_INDEX; index++) {
			printTextFeld(out, index);
		}
		printClosingTag(out, TEXTFELDER);
	}

	public void printZitate(PrintStream out) {
		printOpeningTag(out, ZITATE);
		Iterator<String> it = bogen.getZitate().iterator();
		while (it.hasNext()) {
			printZitat(out, it.next().toString());
		}
		printClosingTag(out, ZITATE);
	}

	public void printLinks(PrintStream out) {
		printOpeningTag(out, LINKS);
		Iterator<Link> it = bogen.getLinks().iterator();
		while (it.hasNext()) {
			printLink(out, it.next());
		}
		printClosingTag(out, LINKS);
	}

	public void printCover(PrintStream out) {
		printOpeningTag(out, COVER);
		out.println(new File(bogen.getCover().toString()).getName());
		printClosingTag(out, COVER);
	}
	
	public void printBMovie(PrintStream out) {
		printOpeningTag(out, BMOVIE);
		printPunktWertungen(out);
		printTextfelder(out);
		printZitate(out);
		printLinks(out);
		printCover(out);
		printClosingTag(out, BMOVIE);
	}

	// /////////////////////////////////////////////////////////////////////////////////////////
	// XML Lesen...

	protected void setzePunktWertung(XMLEvent ev, int index) {
		Iterator<Attribute> attributes = (Iterator<Attribute>) ev
				.asStartElement().getAttributes();
		while (attributes.hasNext()) {
			Attribute attribute = attributes.next();
			if (attribute.getName().toString().equals(PUNKT_VAL)) {
				System.out.println("Fuege punktwert : " + attribute.getValue()
						+ " hinzu");
				bogen.setPunkt(index, Integer.parseInt(attribute.getValue()));
			} else if (attribute.getName().toString().equals(PUNKT_ANMERKUNG)) {
				System.out.println("Fuege anmerkung : " + attribute.getValue()
						+ " hinzu");
				bogen.setAnmerkungen(index, attribute.getValue());
			}
		}

	}

	protected void readPunktwertungen() throws XMLStreamException {
		String tag = "";
		XMLEvent event = null;
		int index = -1;

		System.out.println("Punktwertungen: ");
		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				for (int i = 0; i < Bogen.kategorien.length; i++) {
					if (tag.equals(Bogen.kategorien[i])) {
						index = i;
					}
				}
				if (index > -1) {
					setzePunktWertung(event, index);
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(PUNKTWERTUNGEN)) {
					return;
				} else {
					if (index > -1 && tag.equals(Bogen.kategorien[index])) {
						// nichts zu tun, tag wohlgeformt
						index = -1;
					} else {
						throw new XMLStreamException("    " + tag
								+ " unbekannt");
					}
				}
			}
		}
	}

	protected void readTextfelder() throws XMLStreamException {
		String tag = "", str;
		XMLEvent event = null;
		int index = -1;

		System.out.println("Textfelder:");

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				for (int i = 0; i < Bogen.TEXTFELDER.length; i++) {
					if (tag.equals(Bogen.TEXTFELDER[i])) {
						index = i;
					}
				}
				if (index > -1) {
					str = readText(Bogen.TEXTFELDER[index]);
					System.out.println("Fuege Textfeld "
							+ Bogen.TEXTFELDER[index] + " hinzu");
					System.out.println(str);
					bogen.setText(index, str);
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(TEXTFELDER)) {
					return;
				} else if (index > -1 && tag.equals(Bogen.TEXTFELDER[index])) {
					// nichts zu tun, tag wohlgeformt
					index = -1;
				} else {
					throw new XMLStreamException("   " + tag + " unbekannt");
				}
			}
		}
	}

	protected void readZitate() throws XMLStreamException {
		String tag = "", str;
		XMLEvent event = null;

		System.out.println("Zitate:");

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				if (tag.equals(ZITAT)) {
					str = readText(ZITAT);
					System.out.println("   Fuege Zitat hinzu:");
					System.out.println(str);
					bogen.getZitate().add(str);
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(ZITATE)) {
					return;
				} else {
					throw new XMLStreamException("   " + tag + " unbekannt");
				}
			}
		}
	}

	protected void readLinks() throws XMLStreamException {
		String tag = "", str;
		XMLEvent event = null;
		String typ = "";
		int index = 0;
		
		System.out.println("Links:");

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();
				if (tag.equals(LINK)) {
					Iterator<Attribute> attributes = (Iterator<Attribute>) event
					.asStartElement().getAttributes();
					while (attributes.hasNext()) {
						Attribute attribute = attributes.next();
						if (attribute.getName().toString().equals(LINKS_TYP)) {
							try{
								typ = attribute.getValue();
							}catch(NumberFormatException e){
								System.err.println("readLinks(): Linktyp kein Integer");
								typ = "";
							}
						}
					}
					str = readText(LINK);
					for(index = 0; index < Bogen.LINK_TYPES.length; index++) {
						if(Bogen.LINK_TYPES[index].equals(typ)) {
							System.out.println("   Fuege Link hinzu:");
							System.out.println(str);
							bogen.getLinks().add(new Link(index, str));
						}
					}
				} else {
					throw new XMLStreamException("   Zeile " + tag
							+ " nicht wohlgeformt");
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(LINKS)) {
					return;
				} else {
					throw new XMLStreamException("   " + tag + " unbekannt");
				}
			}
		}
	}
	
	protected String readText(String closingTag) throws XMLStreamException {
		String str = "", tag = "";
		XMLEvent event = null;
		System.out.print("readText: ");

		while (reader.hasNext() && !tag.equals(closingTag)) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				throw new XMLStreamException("Textfeld " + closingTag
						+ " darf keine Untertags enthalten");
			}
			if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (!tag.equals(closingTag)) {
					throw new XMLStreamException("Textfeld " + closingTag
							+ " nicht wohlgeformt");
				}
			}
			if (event.isCharacters()) {
				String np = (event.asCharacters().getData());
				System.out.print(np + "   ");
				str = str + np.replaceAll("[\r\n]", " ").trim();
				// System.out.println("String : " + str);
			}
		}

		System.out.println();

		return str;
	}

	protected void readBMovie() throws XMLStreamException {
		String tag;
		XMLEvent event = null;

		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()) {
				tag = event.asStartElement().getName().getLocalPart();

				if (tag.equals(ZITATE)) {
					readZitate();
				}
				if (tag.equals(PUNKTWERTUNGEN)) {
					readPunktwertungen();
				}
				if (tag.equals(TEXTFELDER)) {
					readTextfelder();
				}
				if (tag.equals(LINKS)) {
					readLinks();
				}
				if (tag.equals(COVER)) {
					bogen.getCover().setS(readText(COVER));
				}
			} else if (event.isEndElement()) {
				tag = event.asEndElement().getName().getLocalPart();
				if (tag.equals(BMOVIE)) {
					return;
				}
			}
		}
	}

	protected Bogen read() throws XMLStreamException {
		XMLEvent event = null;

		// System.out.println(this.bogen);
		while (reader.hasNext()) {
			event = reader.nextEvent();
			if (event.isStartElement()
					&& event.asStartElement().getName().getLocalPart().equals(
							BMOVIE)) {
				readBMovie();
				if (reader.hasNext()) {
					event = reader.nextEvent();
					if (event.isEndDocument()) {
						System.out.println("XML: EndDocument");
					}
				}
				return this.bogen;
			}
		}
		return null;
	}

	public static Bogen readXML(InputStreamReader in, Bogen bogen) {

		XMLInputFactory inputFactory = XMLInputFactory.newInstance();
		try {
			XMLEventReader eventReader = inputFactory.createXMLEventReader(in);
			XML xml = new XML(eventReader, bogen);
			return xml.read();

		} catch (XMLStreamException e) {
			System.err.println("XML::readXML: Fehler mit XMLStream: "
					+ e.toString());
		}
		return null;
	}
}
