/**CFile*******************************************************************
  PackageName [Bdd Scout]
  Synopsis    [Bdd Scout]

  FileName    [bddscout.c]
  Revision    [$Revision: 30 $]
  Date        [$Date: 2013-04-05 15:35:44 +0200 (pet, 05 apr 2013) $]
  Authors     [Robert Meolic (meolic@uni-mb.si)]
  Description []
  SeeAlso     [bddscout.h]

  Copyright   [This file is part of Bdd Scout package.
               Copyright (C) 2008, 2013 UM-FERI
               UM-FERI, Smetanova ulica 17, SI-2000 Maribor, Slovenia

               Bdd Scout is free software; you can redistribute it and/or modify
               it under the terms of the GNU General Public License as
               published by the Free Software Foundation; either version 2
               of the License, or (at your option) any later version.

               Bdd Scout is distributed in the hope that it will be useful,
               but WITHOUT ANY WARRANTY; without even the implied warranty of
               MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
               GNU General Public License for more details.

               You should have received a copy of the GNU General Public
               License along with this program; if not, write to the Free
               Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
               Boston, MA 02110-1301 USA.]
  ************************************************************************/

#include "bddscout.h"

#ifdef UNIX
#include <sys/wait.h>
#endif

/* used for STUBS mechanism */
extern BddscoutStubs bddscoutStubs;

/* on tcl 8.3 use #define USECONST */
/* on tcl 8.4 use #define USECONST const*/
/* this is defined in Makefile */

/*-----------------------------------------------------------------------*/
/* Type declarations                                                     */
/*-----------------------------------------------------------------------*/

/*-----------------------------------------------------------------------*/
/* Structure declarations                                                */
/*-----------------------------------------------------------------------*/

typedef struct BddscoutFormula {
  struct BddscoutFormula *l, *r;
  Biddy_Edge f;
  Biddy_String name;
} BddscoutFormula;

/*-----------------------------------------------------------------------*/
/* Variable declarations                                                 */
/*-----------------------------------------------------------------------*/

BddscoutFormula *bddscoutFormulaeTree = NULL;
int SCAN_RESULT; /* used in readln macro */

/*-----------------------------------------------------------------------*/
/* External functions                                                    */
/*-----------------------------------------------------------------------*/

/**Function****************************************************************
  Synopsis    [Function Bddscout_FindFormula]
  Description [Bddscout_FindFormula find formula in Fomulae tree.]
  SideEffects []
  SeeAlso     [Bddscout_AddFormula]
  ************************************************************************/

Biddy_Boolean
Bddscout_FindFormula(Biddy_String x, Biddy_Edge *f)
{
  BddscoutFormula *sup;

  sup = bddscoutFormulaeTree;
  while (sup && strcmp(sup->name, x)) {
    sup = (strcmp(x, sup->name) < 0) ? sup->l : sup->r;
  }
  if (!sup) {
    *f = biddy_null;
    return FALSE;
  } else {
    *f = sup->f;
    return TRUE;
  }
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_AddFormula]
  Description [Bddscout_AddFormula add formula to the Formulae tree.]
  SideEffects [If bddscoutFormulaeTree does not exist, function create it.
               If such formula already exists, function overwrite it!]
  SeeAlso     [Bddscout_FindFormula]
  ************************************************************************/

void
Bddscout_AddFormula(Biddy_String x, Biddy_Edge f)
{
  BddscoutFormula *sup;
  Biddy_Boolean end, overwrite;

  overwrite = FALSE;
  if (!bddscoutFormulaeTree) {

    if (!(bddscoutFormulaeTree = (BddscoutFormula *) malloc(sizeof(BddscoutFormula)))) {
      fprintf(stdout,"Bddscout_AddFormula: Out of memoy!\n");
      exit(1);
    }

    bddscoutFormulaeTree->l = bddscoutFormulaeTree->r = NULL;
    bddscoutFormulaeTree->name = strdup(x);
    bddscoutFormulaeTree->f = f;
  } else {
    end = FALSE;
    sup = bddscoutFormulaeTree;
    while (!end) {
      if (strcmp(x, sup->name) < 0) {
        if (sup->l) sup = sup->l;
          else end = TRUE;
      } else {
        if (strcmp(x, sup->name) > 0) {
          if (sup->r) sup = sup->r;
            else end = TRUE;
        } else {
          sup->f = f;
          overwrite = TRUE;
          end = TRUE;
        }
      }
    }

    if (!overwrite) {
      if (strcmp(x, sup->name) < 0) {

        if (!(sup->l = (BddscoutFormula *) malloc(sizeof(BddscoutFormula)))) {
          fprintf(stdout,"Bddscout_AddFormula: Out of memoy!\n");
          exit(1);
        }

        sup = sup->l;

      } else {

        if (!(sup->r = (BddscoutFormula *) malloc(sizeof(BddscoutFormula)))) {
          fprintf(stdout,"Bddscout_AddFormula: Out of memoy!\n");
          exit(1);
        }

        sup = sup->r;
      }
      sup->l = sup->r = NULL;
      sup->name = strdup(x);
      sup->f = f;

    }
  }
}

/*-----------------------------------------------------------------------*/
/* Internal functions                                                    */
/*-----------------------------------------------------------------------*/

/**Function****************************************************************
  Synopsis    [Function Bddscout_DeleteFormulaeTree]
  Description []
  SideEffects []
  SeeAlso     []
  ************************************************************************/

void
Bddscout_DeleteFormulaeTree(BddscoutFormula *f)
{
  if (f != NULL) {
    Bddscout_DeleteFormulaeTree(f->l);
    Bddscout_DeleteFormulaeTree(f->r);
    free(f->name);
    free(f);
  }
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ListFormulaByName]
  Description [Bddscout_ListFormulaByName creates a sorted list of all
               formulae in Fomulea tree.]
  SideEffects []
  SeeAlso     []
  ************************************************************************/

void
Bddscout_ListFormulaByName(Biddy_String *list, BddscoutFormula *t)
{
  Biddy_String newlist;

  if (!t) return;

  if (t->l) Bddscout_ListFormulaByName(list, t->l);

  newlist = (Biddy_String) malloc(strlen(*list)+strlen(t->name)+2);
  sprintf(newlist,"%s %s",*list,t->name);
  free(*list);
  *list = newlist;

  if (t->r) Bddscout_ListFormulaByName(list, t->r);
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ListFormulaByNodeNumber]
  Description [Bddscout_ListFormulaByNodeNumber creates a sorted list of all
               formulae in Fomulea tree.]
  SideEffects []
  SeeAlso     []
  ************************************************************************/

void
Bddscout_ListFormulaByNodeNumber(Biddy_String *list, BddscoutFormula *t)
{
  typedef struct {
    BddscoutFormula *t;
    int nodeNumber;
  } FT;

  static FT *sortingTable;
  static int tableSize;
  Biddy_String newlist;
  Biddy_String number;
  int i;
  int n;

  if (!t) return;
  if (!strcmp(*list,"NULL")) {

    free(*list);
    *list = strdup("");
    sortingTable = (FT *) malloc(sizeof(FT));
    tableSize = 1;
    sortingTable[0].t = t;
    sortingTable[0].nodeNumber = Biddy_NodeNumber(t->f);

    if (t->l) Bddscout_ListFormulaByNodeNumber(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeNumber(list,t->r);

    for (i=0;i<tableSize;++i) {
      number = (Biddy_String) malloc(255);
      sprintf(number,"(%d)",sortingTable[i].nodeNumber);
      newlist = (Biddy_String)
        malloc(strlen(*list)+ strlen(sortingTable[i].t->name)+strlen(number)+2);
      sprintf(newlist,"%s %s%s",*list,sortingTable[i].t->name,number);
      free(number);
      free(*list);
      *list = newlist;
    }

  } else {

    tableSize++;
    sortingTable = (FT *) realloc(sortingTable,sizeof(FT)*tableSize);

    i = tableSize-2;
    n = Biddy_NodeNumber(t->f);
    while ((i>=0) && (sortingTable[i].nodeNumber > n)) {
      sortingTable[i+1].t = sortingTable[i].t;
      sortingTable[i+1].nodeNumber = sortingTable[i].nodeNumber;
      i--;
    }
    sortingTable[i+1].t = t;
    sortingTable[i+1].nodeNumber = n;

    if (t->l) Bddscout_ListFormulaByNodeNumber(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeNumber(list,t->r);
  }
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ListFormulaByNodeMaxLevel]
  Description [Bddscout_ListFormulaByNodeMaxLevel creates a sorted list of all
               formulae in Fomulea tree.]
  SideEffects []
  SeeAlso     []
  ************************************************************************/

void
Bddscout_ListFormulaByNodeMaxLevel(Biddy_String *list, BddscoutFormula *t)
{
  typedef struct {
    BddscoutFormula *t;
    int nodeMaxLevel;
  } FT;

  static FT *sortingTable;
  static int tableSize;
  Biddy_String newlist;
  Biddy_String number;
  int i;
  int n;

  if (!t) return;
  if (!strcmp(*list,"NULL")) {

    free(*list);
    *list = strdup("");
    sortingTable = (FT *) malloc(sizeof(FT));
    tableSize = 1;
    sortingTable[0].t = t;
    sortingTable[0].nodeMaxLevel = Biddy_NodeMaxLevel(t->f);

    if (t->l) Bddscout_ListFormulaByNodeMaxLevel(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeMaxLevel(list,t->r);

    for (i=0;i<tableSize;++i) {
      number = (Biddy_String) malloc(255);
      sprintf(number,"(%d)",sortingTable[i].nodeMaxLevel);
      newlist = (Biddy_String)
        malloc(strlen(*list)+ strlen(sortingTable[i].t->name)+strlen(number)+2);
      sprintf(newlist,"%s %s%s",*list,sortingTable[i].t->name,number);
      free(number);
      free(*list);
      *list = newlist;
    }

  } else {

    tableSize++;
    sortingTable = (FT *) realloc(sortingTable,sizeof(FT)*tableSize);

    i = tableSize-2;
    n = Biddy_NodeMaxLevel(t->f);
    while ((i>=0) && (sortingTable[i].nodeMaxLevel > n)) {
      sortingTable[i+1].t = sortingTable[i].t;
      sortingTable[i+1].nodeMaxLevel = sortingTable[i].nodeMaxLevel;
      i--;
    }
    sortingTable[i+1].t = t;
    sortingTable[i+1].nodeMaxLevel = n;

    if (t->l) Bddscout_ListFormulaByNodeMaxLevel(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeMaxLevel(list,t->r);
  }
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ListFormulaByNodeAvgLevel]
  Description [Bddscout_ListFormulaByNodeAvgLevel creates a sorted list of all
               formulae in Fomulea tree.]
  SideEffects []
  SeeAlso     []
  ************************************************************************/

void
Bddscout_ListFormulaByNodeAvgLevel(Biddy_String *list, BddscoutFormula *t)
{
  typedef struct {
    BddscoutFormula *t;
    float nodeAvgLevel;
  } FT;

  static FT *sortingTable;
  static int tableSize;
  Biddy_String newlist;
  Biddy_String number;
  int i;
  float n;

  if (!t) return;
  if (!strcmp(*list,"NULL")) {

    free(*list);
    *list = strdup("");
    sortingTable = (FT *) malloc(sizeof(FT));
    tableSize = 1;
    sortingTable[0].t = t;
    sortingTable[0].nodeAvgLevel = Biddy_NodeAvgLevel(t->f);

    if (t->l) Bddscout_ListFormulaByNodeAvgLevel(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeAvgLevel(list,t->r);

    for (i=0;i<tableSize;++i) {
      number = (Biddy_String) malloc(255);
      sprintf(number,"(%.2f)",sortingTable[i].nodeAvgLevel);
      newlist = (Biddy_String)
        malloc(strlen(*list)+ strlen(sortingTable[i].t->name)+strlen(number)+2);
      sprintf(newlist,"%s %s%s",*list,sortingTable[i].t->name,number);
      free(number);
      free(*list);
      *list = newlist;
    }

  } else {

    tableSize++;
    sortingTable = (FT *) realloc(sortingTable,sizeof(FT)*tableSize);

    i = tableSize-2;
    n = Biddy_NodeAvgLevel(t->f);
    while ((i>=0) && (sortingTable[i].nodeAvgLevel > n)) {
      sortingTable[i+1].t = sortingTable[i].t;
      sortingTable[i+1].nodeAvgLevel = sortingTable[i].nodeAvgLevel;
      i--;
    }
    sortingTable[i+1].t = t;
    sortingTable[i+1].nodeAvgLevel = n;

    if (t->l) Bddscout_ListFormulaByNodeAvgLevel(list,t->l);
    if (t->r) Bddscout_ListFormulaByNodeAvgLevel(list,t->r);
  }
}

/*-----------------------------------------------------------------------*/
/* Other internal functions (mostly from 1995)                           */
/*-----------------------------------------------------------------------*/

/**Function****************************************************************
  Synopsis    [Function Bddscout_ParseString]
  Description []
  SideEffects [Implemented long time ago (1995). Not comprehensible.]
  SeeAlso     []
  ************************************************************************/

#define oAND 1
#define oOR 2
#define oEXOR 3

static void nextCh(Biddy_String s, int *i, Biddy_String *ch);
static Biddy_Edge evaluate1(Biddy_String s, int *i, Biddy_String *ch);
static Biddy_Edge evaluateN(Biddy_String s, int *i, Biddy_String *ch,
                            Biddy_Edge g, int op);
static int Op(Biddy_String s, int *i, Biddy_String *ch);
static Biddy_Boolean charOK(char c);

Biddy_Edge
Bddscout_ParseString(Biddy_String s)
{
  Biddy_Edge sup;
  Biddy_String ch;
  int i;

  ch = (Biddy_String) malloc(255);

  i = 0;
  nextCh(s,&i,&ch);
  sup = evaluate1(s,&i,&ch);

  free(ch);
  return sup;
}

static void
nextCh(Biddy_String s, int *i, Biddy_String *ch)
{
  char c;
  int j;

  while (s[*i] == ' ' || s[*i] == '\t' || s[*i] == '\n') (*i)++;
  j = *i;
  c = s[*i];

  if (c == '(') {
    (*i)++;
    strcpy(*ch,"(");
    return;
  }

  if (c == ')') {
    (*i)++;
    strcpy(*ch,")");
    return;
  }

  if (charOK(c)) {
    (*i)++;
    while (charOK(s[*i])) (*i)++;
    c = s[*i];

    if ( (c == ' ') || (c == '\t') || (c == '\n') || (c == '(') || (c == ')') ) {
      strncpy(*ch, &(s[j]), *i-j);
      (*ch)[*i-j] = 0;
      return;
    }
  }

  strcpy(*ch,"");
  return;
}

static Biddy_Edge
evaluate1(Biddy_String s, int *i, Biddy_String *ch)
{
  Biddy_Edge f;
  int op;

  if (!strcmp(*ch,"(")) {
    nextCh(s,i,ch);

    if (!strcasecmp(*ch, "NOT")) {
      nextCh(s,i,ch);
      f = Biddy_NOT(evaluate1(s,i,ch));
    } else {
      if ((op = Op(s,i,ch))) {
        f = evaluateN(s, i, ch, evaluate1(s,i,ch), op);
      } else {
        f = evaluate1(s,i,ch);
      }
    }

    if (!strcmp(*ch, ")")) nextCh(s,i,ch);

    return f;
  }

  if (charOK(*ch[0])) {
    if (!Bddscout_FindFormula(*ch, &f)) f = Biddy_FoaTerminal(*ch);
    nextCh(s,i,ch);
    return f;
  }

  return biddy_null;
}

static Biddy_Edge
evaluateN(Biddy_String s, int *i, Biddy_String *ch, Biddy_Edge g, int op)
{
  Biddy_Edge f, h;

  h = evaluate1(s,i,ch);
  if (!Biddy_isNull(h)) {
    switch (op) {
            case oAND:  f = Biddy_ITE(g, h, biddy_zero);
                        break;
            case oOR:  f = Biddy_ITE(g, biddy_one, h);
                        break;
            case oEXOR:  f = Biddy_ITE(g, Biddy_NOT(h), h);
    }
    return evaluateN(s, i, ch, f, op);
  }
  return g;
}

static int
Op(Biddy_String s, int *i, Biddy_String *ch)
{
  if (!strcasecmp(*ch, "AND")) {
    nextCh(s,i,ch);
    return oAND;
  }

  if (!strcasecmp(*ch, "OR")) {
    nextCh(s,i,ch);
    return oOR;
  }

  if (!strcasecmp(*ch, "EXOR")) {
    nextCh(s,i,ch);
    return oEXOR;
  }

  return 0;
}

static Biddy_Boolean
charOK(char c)
{
  return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || c == '_' ||
         (c >= '0' && c <= '9') || c == '[' || c == ']' ||
         c == '+' || c == '-' || c == '*' || c == '\'' ||
         c == '$' || c == '&' || c == '%' || c == '#' ||
         c == '?' || c == '!' || c == ':' || c == '.' ;
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ParseFile]
  Description []
  SideEffects [Implemented long time ago (1995). Not comprehensible.]
  SeeAlso     []
  ************************************************************************/

#define readln(f,s) SCAN_RESULT = fscanf(f," %[^\n] ",s)

static void makeFunction(FILE *f, Biddy_String line);

void
Bddscout_ParseFile(FILE *f)
{
  Biddy_String line;
  Biddy_String s1;
  Biddy_String var;
  Biddy_Edge tmp;
  unsigned int n,x;

  line = (Biddy_String) malloc(4096);

  readln(f,line);
  x = strcspn(line,"=");
  if (x != strlen(line)) {
    printf("WARNING: Variable ordering not specified.\n");
  } else {
    s1 = strdup(line);
    readln(f,line);
    x = strcspn(line,"=");
    while (x == strlen(line))
    {
      n = strlen(s1)+strlen(line)+1+1;
      s1 = (Biddy_String) realloc(s1,n);
      strcat(s1," ");
      strcat(s1,line);
      readln(f,line);
      x = strcspn(line,"=");
    }

    var = strtok(&s1[1]," \t)");
    while (var) {

      /*
      printf("VARIABLE: <%s>\n",var);
      */

      if (Bddscout_FindFormula(var,&tmp)) {
        printf("Warning: Variable %s will not be used because Boolean function with the same name already exists.\n",var);
      } else {
        Biddy_FoaTerminal(var);
      }
      var = strtok(NULL," \t)");
    }

    free(s1);
  }

  while (!feof(f)) {
    makeFunction(f,line);
  }
  makeFunction(f,line);

  free(line);

}

static void
makeFunction(FILE *f, Biddy_String line)
{
  Biddy_Edge fun;
  Biddy_String s1,s2,name;
  unsigned int len;
  unsigned int x,y;

  if (!line) return;

  x = strcspn(line, "=");
  if (x == strlen(line)) {
    s1 = NULL;                 /* s1 = left part of the expression */
    s2 = strdup(line);         /* s2 = right part of the expression */
  } else {
    s1 = (Biddy_String) strdup(line);
    s1[x] = 0; /* strndup is not supported everywhere */
    if (x != strlen(line)-1) {
      y = strspn(&line[x+1], " \t");
      s2 = strdup(&line[x+1+y]);
    } else {
      s2 = NULL;
    }
  }

  if (s1) {
    x = strspn(s1," \t");
    y = strcspn(&s1[x]," \t\n");
    name = (Biddy_String) strdup(&s1[x]);
    name[y] = 0; /* strndup is not supported everywhere */
    free(s1);
  } else {
    name = strdup("F");
  }

  if (!feof(f)) {
    readln(f,line);
    x = strcspn(line,"=");
  } else {
    line[0] = 0;
    line = NULL;
  }

  while (line && (x == strlen(line)))
  {
    if (s2 != NULL) {
      len = strlen(s2)+strlen(line)+1;
      s2 = (Biddy_String) realloc(s2, len+1);
      strcat(s2," ");
      strcat(s2,line);
    } else {
      s2 = strdup(line);
    }

    if (!feof(f)) {
      readln(f,line);
      x = strcspn(line, "=");
    } else {
      line[0] = 0;
      line = NULL;
    }
  }

  /*
  printf("FUNCTION NAME: <%s>\n",name);
  printf("FUNCTION DEF: <%s>\n",s2);
  */

  fun = Bddscout_ParseString(s2);
  free(s2);

  if (!Biddy_isNull(fun)) {


    /* USE THIS TO ENABLE GARBAGE COLLECTION */
    /**/
    Biddy_Fortify(fun);
    Biddy_IncCounter();
    /**/

    Bddscout_AddFormula(name,fun);

  }

  free(name);
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_WriteDot.]
  Description []
  SideEffects [if (id != -1) use it instead of <...>]
  SeeAlso     []
  ************************************************************************/

typedef struct {
  int n;
  void *p;
  int t;
} tNode;

static void EnumerateNodes(Biddy_Edge f, tNode **tnode, int *t, int *n);
static void WriteDot_Nodes(FILE *dotfile, tNode *tnode, int n, int id);
static void WriteDot_Edges(FILE *dotfile, Biddy_Edge f, tNode *tnode, int *t, int n);
static void addNode(tNode **tnode, int n, void *p, int t);
static int findNode(tNode *tnode, int n, void *p, int t);
static Biddy_String getvariablename(void *p);
static Biddy_String getname(void *p);
static Biddy_String getshortname(void *p, int n);

int
Bddscout_WriteDot(FILE *s, Biddy_String name, int id)
{
  Biddy_Edge f;
  int t,n;
  tNode *tn;
  Biddy_String hash;

  n = 0;
  if (!Bddscout_FindFormula(name,&f)) {
    printf("Function %s does not exists!\n",name);
    return(0);
  } else {

    while ( (hash=strchr(name,'#')) ) hash[0]='_';

    fprintf(s,"digraph BDD {\n");
    fprintf(s,"  ordering = out;\n");
    fprintf(s,"  splines = true;\n");
    fprintf(s,"  node [shape = none, label = \"%s\"] 0;\n",name);

    t = 0;
    n = 0;
    tn = NULL;
    EnumerateNodes(f,&tn,&t,&n);
    Biddy_NodeRepair(f);
    WriteDot_Nodes(s,tn,n,id);
    fprintf(s,"  0 -> 1;\n");
    t = 1;
    WriteDot_Edges(s,f,tn,&t,n);
    Biddy_NodeRepair(f);
    fprintf(s,"}\n");

    free(tn);
  }

  return n;
}

static void
EnumerateNodes(Biddy_Edge f, tNode **tnode, int *t, int *n)
{
  if (!Biddy_isSelected(f)) {
    Biddy_SelectNode(f);
    if (Biddy_isTerminal(f)) {
      (*n)++;
      (*t)++;
      addNode(tnode,(*n),Biddy_GetPointer(f),(*t));
    } else {
      (*n)++;
      addNode(tnode,(*n),Biddy_GetPointer(f),0);
      if (Biddy_isTerminal(Biddy_GetElse(f)) || Biddy_isTerminal(Biddy_GetThen(f))) {
        (*n)++;
        (*t)++;
        addNode(tnode,(*n),Biddy_GetPointer(f),(*t));
      }
      if (!Biddy_isTerminal(Biddy_GetElse(f))) {
        EnumerateNodes(Biddy_GetElse(f),tnode,t,n);
      }
      if (!Biddy_isTerminal(Biddy_GetThen(f))) {
        EnumerateNodes(Biddy_GetThen(f),tnode,t,n);
      }
    }
  }
}

static void
WriteDot_Nodes(FILE *dotfile, tNode *tnode, int n, int id)
{
  int i;
  void *p;
  Biddy_String name,hash;

  for (i=0; i<n; ++i) {
    if (tnode[i].t == 0) {
      p = tnode[i].p;
      if (id == -1) {
        name = getname(p);
      } else {
        name = getshortname(p,id);
      }
      while ( (hash=strchr(name,'#')) ) hash[0]='_';

      fprintf(dotfile,"  node [shape = circle, label = \"%s\"] %d;\n",name,tnode[i].n);
      free(name);
    } else {
      fprintf(dotfile,"  node [shape = none, label = \"1\"] %d;\n",tnode[i].n);
    }
  }
}

static void
WriteDot_Edges(FILE *dotfile, Biddy_Edge f, tNode *tnode, int *t, int n)
{
  int n1,n2;

  if (!Biddy_isTerminal(f) && !Biddy_isSelected(f)) {
    Biddy_SelectNode(f);
    n1 = findNode(tnode,n,Biddy_GetPointer(f),0);

    if (Biddy_isTerminal(Biddy_GetElse(f))) {
      n2 = findNode(tnode,n,Biddy_GetPointer(f),(*t));
    } else {
      n2 = findNode(tnode,n,Biddy_GetPointer(Biddy_GetElse(f)),0);
    }
    fprintf(dotfile,"  %d -> %d;\n",n1,n2);

    if (Biddy_isTerminal(Biddy_GetThen(f))) {
      n2 = findNode(tnode,n,Biddy_GetPointer(f),(*t));
    } else {
      n2 = findNode(tnode,n,Biddy_GetPointer(Biddy_GetThen(f)),0);
    }
    fprintf(dotfile,"  %d -> %d;\n",n1,n2);

    if (Biddy_isTerminal(Biddy_GetElse(f)) || Biddy_isTerminal(Biddy_GetThen(f))) (*t)++;
    WriteDot_Edges(dotfile,Biddy_GetElse(f),tnode,t,n);
    WriteDot_Edges(dotfile,Biddy_GetThen(f),tnode,t,n);
  }
}

static void
addNode(tNode **tnode, int n, void *p, int t)
{
  if (!(*tnode)) {
    (*tnode) = (tNode *) malloc(n * sizeof(tNode));
  } else {
    (*tnode) = (tNode *) realloc((*tnode), n * sizeof(tNode));
  }
  (*tnode)[n-1].n = n;
  (*tnode)[n-1].p = p;
  (*tnode)[n-1].t = t;
}

static int
findNode(tNode *tnode, int n, void *p, int t)
{
  int i;

  i=0;
  while (i<n) {
    if ((tnode[i].p == p) && (tnode[i].t == t)) {
      return tnode[i].n;
    }
    i++;
  }

  return -1;
}

static Biddy_String
getvariablename(void *p) {
  Biddy_Edge tmp;

  /* tmp = biddy_null; */
  tmp = p; /*NEW: tmp.p => tmp*/
  return (Biddy_GetVariableName(tmp));
}

static Biddy_String
getname(void *p) {
  unsigned int i;
  Biddy_String newname;

  newname = strdup(getvariablename(p));
  for (i=0; i<strlen(newname); ++i) {
    if (newname[i] == '<') newname[i] = '_';
    if (newname[i] == '>') newname[i] = 0;
  }

  return (newname);
}

static Biddy_String
getshortname(void *p, int n) {
  int i;
  Biddy_String name;
  Biddy_String shortname;

  name = strdup(getvariablename(p));
  i = strcspn(name,"<");
  name[i]=0;
  shortname = strdup(name);
  free(name);

  return (shortname);
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_WriteBDDView.]
  Description []
  SideEffects [Uses Bddscout_WriteDot and graphviz package]
  SeeAlso     []
  ************************************************************************/

typedef struct {
  int id;
  Biddy_String label;
  int x;
  int y;
} tableXY;

static void WriteDot_Connections(FILE *funfile, Biddy_Edge f, tNode *tnode,
                                 int *t, int n);
static int ParseDot(FILE *dotfile, tableXY *table, int *xsize, int *ysize);

int
Bddscout_WriteBDDView(FILE *s, Biddy_String dotexe, Biddy_String name, int id)
{
  char dotname[L_tmpnam];
  char xdotname[L_tmpnam];
  FILE *dotfile;
  FILE *xdotfile;
  int i,t,n;
  tNode *tn;
  tableXY *table;
  int xsize,ysize;
  Biddy_Edge f;
#ifndef WIN32
  pid_t childpid;
  int status;
#endif

  if (!Bddscout_FindFormula(name,&f)) {
    printf("Function %s does not exists!\n",name);
    return -1;
  } else {

#ifdef WIN32
    strcpy(dotname,"bdd.dot");
    strcpy(xdotname,"bdd.xdot");
#else
    if (!tmpnam(dotname)) {
      printf("ERROR: tmpnam(dotname)!\n");
      return -1;
    }
    if (!tmpnam(xdotname)) {
      printf("ERROR: tmpnam(xdotname)!\n");
      return -1;
    }
#endif

    dotfile = fopen(dotname,"w");
    if (!dotfile) {
      printf("Bddscout_WriteBDDView: File error (%s)!\n",dotname);
      return -1;
    }

    n = Bddscout_WriteDot(dotfile,name,id); /* n = number of nodes */
    fclose(dotfile);

#ifdef WIN32
    _spawnlp(_P_WAIT, dotexe, "dot.exe", "-y", "-o", xdotname, dotname, (char *) NULL);
#else
    childpid = fork();
    if (childpid == 0) {
      execlp(dotexe, "dot", "-y", "-o", xdotname, dotname, (char *) NULL);
    } else if (childpid > 0) {
      waitpid(childpid,&status,0);
    }
#endif

    xdotfile = fopen(xdotname,"r");
    if (!xdotfile) {
      printf("Bddscout_WriteBDDView: File error (%s)!\n",xdotname);
      return -1;
    }

    table = (tableXY *) malloc((n+1) * sizeof(tableXY)); /* n = nodes + label */
    ParseDot(xdotfile,table,&xsize,&ysize);
    fclose(xdotfile);

    t = 0;
    n = 0;
    tn = NULL;
    EnumerateNodes(f,&tn,&t,&n);
    Biddy_NodeRepair(f);

    fprintf(s,"label %d %s %d %d\n",table[0].id,table[0].label,table[0].x,table[0].y);
    for (i=1;i<(n+1);++i) {
      if (tn[i-1].t == 0) {
        fprintf(s,"node %d %s %d %d\n",table[i].id,table[i].label,table[i].x,table[i].y);
      } else {
        fprintf(s,"terminal %d %s %d %d\n",table[i].id,table[i].label,table[i].x,table[i].y);
      }
      free(table[i].label);
    }
    free(table);

    fprintf(s,"connect 0 1 ");
    if (Biddy_GetMark(f)) {
      fprintf(s,"si\n");
    } else {
      fprintf(s,"s\n");
    }

    t = 1;
    WriteDot_Connections(s,f,tn,&t,n);
    Biddy_NodeRepair(f);

    free(tn);

  }

  return 0;
}

static void
WriteDot_Connections(FILE *funfile, Biddy_Edge f, tNode *tnode, int *t, int n)
{
  int n1,n2;

  if (!Biddy_isTerminal(f) && !Biddy_isSelected(f)) {
    Biddy_SelectNode(f);
    n1 = findNode(tnode,n,Biddy_GetPointer(f),0);

    if (Biddy_GetPointer(Biddy_GetThen(f)) == Biddy_GetPointer(Biddy_GetElse(f))) {

      if (Biddy_isTerminal(Biddy_GetThen(f))) {
        n2 = findNode(tnode,n,Biddy_GetPointer(f),(*t));
      } else {
        n2 = findNode(tnode,n,Biddy_GetPointer(Biddy_GetThen(f)),0);
      }
      fprintf(funfile,"connect %d %d d\n",n1,n2);

    } else {

      if (Biddy_isTerminal(Biddy_GetElse(f))) {
        n2 = findNode(tnode,n,Biddy_GetPointer(f),(*t));
      } else {
        n2 = findNode(tnode,n,Biddy_GetPointer(Biddy_GetElse(f)),0);
      }
      fprintf(funfile,"connect %d %d ",n1,n2);
      if (Biddy_GetMark((Biddy_GetElse(f)))) {
        fprintf(funfile,"li\n");
      } else {
        fprintf(funfile,"l\n");
      }

      if (Biddy_isTerminal(Biddy_GetThen(f))) {
        n2 = findNode(tnode,n,Biddy_GetPointer(f),(*t));
      } else {
        n2 = findNode(tnode,n,Biddy_GetPointer(Biddy_GetThen(f)),0);
      }
      fprintf(funfile,"connect %d %d r\n",n1,n2);

    }

    if (Biddy_isTerminal(Biddy_GetElse(f)) || Biddy_isTerminal(Biddy_GetThen(f))) (*t)++;
    WriteDot_Connections(funfile,Biddy_GetElse(f),tnode,t,n);
    WriteDot_Connections(funfile,Biddy_GetThen(f),tnode,t,n);
  }
}

static int
ParseDot(FILE *dotfile, tableXY *table, int *xsize, int *ysize)
{
  char line[128];
  Biddy_String r,label,pos;
  int n,id,x,y;

  n = 0;

  r = fgets(line,128,dotfile);
  while (r) {
    if (strstr(line,"bb=")) {
      pos = (Biddy_String) malloc(strlen(line));
      sscanf(&(strstr(line,"bb=")[4]),"%s",pos);
      sscanf(pos,"%d,%d,%d,%d",&x,&y,xsize,ysize);
      free(pos);
    }
    if ((strstr(line,"label=")) && (strstr(line,"pos="))) {

      /*
      fprintf(stderr,line);
      */

      label = (Biddy_String) malloc(strlen(line));
      pos = (Biddy_String) malloc(strlen(line));

      sscanf(line,"%d",&id);
      sscanf(&(strstr(line,"label=")[6]),"%s",label);
      label[strlen(label)-1] = 0;
      sscanf(&(strstr(line,"pos=")[5]),"%s",pos);
      sscanf(pos,"%d,%d",&x,&y);

      /*
      fprintf(stderr,"id = %d, label = %s, x = %d, y = %d\n",id,label,x,y);
      */

      table[n].id = id;
      table[n].label = strdup(label);
      table[n].x = x;
      table[n].y = y;

      n++;
      free(label);
      free(pos);
    }

    while (r &&
           !(strchr(line,';')) &&
           !(strchr(line,'{')) &&
           !(strchr(line,'}'))) r= fgets(line,128,dotfile);

    if (r) r = fgets(line,128,dotfile);
  }

  return (n-1);
}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ConstructBDD.]
  Description []
  SideEffects []
  SeeAlso     []
  ************************************************************************/

typedef struct {
  Biddy_String name;
  int order;
} BddscoutVariableOrder;

typedef struct {
  Biddy_String name;
  int type;
  int l,r;
  Biddy_Edge f;
  Biddy_Boolean created;
} BddscoutTclBDD;

static void reorderVariables(BddscoutVariableOrder *tableV,
			  BddscoutTclBDD *tableN, int id);

static void constructBDD(BddscoutTclBDD *table, int id);

void
Bddscout_ConstructBDD(int numV, Biddy_String s2, int numN, Biddy_String s4)
{
  BddscoutVariableOrder *tableV;
  BddscoutTclBDD *tableN;
  Biddy_String word;
  int i,id;

  /*
  printf("Construct BDD\n");
  printf("%d variables\n",numV);
  printf("%s\n",s2);
  printf("%d nodes\n",numN);
  printf("%s\n",s4);
  */

  tableV = (BddscoutVariableOrder *) malloc(numV * sizeof(BddscoutVariableOrder));
  tableN = (BddscoutTclBDD *) malloc(numN * sizeof(BddscoutTclBDD));

  if (numV) {
    word = strtok(s2," "); /* parsing list of variables */
    tableV[0].name = strdup(word);
    tableV[0].order = 0;
    for (i=1; i<numV; ++i) {
      word = strtok(NULL," ");
      tableV[i].name = strdup(word);
      tableV[i].order = 0;
    }
  }

  /* type == 0 : regular label, r = -1 */
  /* type == 1 : complemented label, r = -1 */
  /* type == 2 : terminal node, l = -1, r = -1 */
  /* type == 3 : regular node */
  /* type == 4 : node with complemented "else" */

  word = strtok(s4," "); /* the name of BDD */
  for (i=0; i<numN; ++i) {
    word = strtok(NULL," "); /* node id */
    id = atoi(word);
    word = strtok(NULL," ");
    tableN[id].name = strdup(word);
    word = strtok(NULL," ");
    tableN[id].type = atoi(word);
    word = strtok(NULL," ");
    tableN[id].l = atoi(word);
    word = strtok(NULL," ");
    tableN[id].r = atoi(word);

    if (tableN[id].type == 2) {
      tableN[id].created = TRUE;
      tableN[id].f = biddy_one;
    } else {
      tableN[id].created = FALSE;
      tableN[id].f = biddy_null;
    }
  }

  for (i=0; i<numV; ++i) {
    Biddy_FoaTerminal(tableV[i].name);
  }

  for (i=0; i<numN; ++i) {
    if (tableN[i].type == 0) {
      reorderVariables(tableV,tableN,tableN[i].l);
      constructBDD(tableN,tableN[i].l);
      Bddscout_AddFormula(tableN[i].name,tableN[tableN[i].l].f);
    }
    if (tableN[i].type == 1) {
      reorderVariables(tableV,tableN,tableN[i].l);
      constructBDD(tableN,tableN[i].l);
      Bddscout_AddFormula(tableN[i].name,Biddy_NOT(tableN[tableN[i].l].f));
    }
  }

  for (i=0; i<numV; ++i) {
    free(tableV[i].name);
  }
  for (i=0; i<numN; ++i) {
    free(tableN[i].name);
  }

  free(tableV);
  free(tableN);
}

static void
reorderVariables(BddscoutVariableOrder *tableV, BddscoutTclBDD *tableN, int id)
{
  /* NOT IMPLEMENTED, YET */
}

static void
constructBDD(BddscoutTclBDD *tableN, int id)
{
  /*
  printf("Construct %d\n",id);
  printf("TABLE[%d]: %s, l:%d, r:%d\n",
          id,tableN[id].name,tableN[id].l,tableN[id].r);
  */

  if (!tableN[id].created) {

    if (!tableN[tableN[id].l].created) {
      constructBDD(tableN,tableN[id].l);
    }

    if ((tableN[id].r != -1) && (!tableN[tableN[id].r].created)) {
      constructBDD(tableN,tableN[id].r);
    }

    if (tableN[id].type == 3) {
      tableN[id].f = Biddy_ITE(Biddy_FoaTerminal(tableN[id].name),
                              tableN[tableN[id].r].f,tableN[tableN[id].l].f);
    }

    if (tableN[id].type == 4) {
      tableN[id].f = Biddy_ITE(Biddy_FoaTerminal(tableN[id].name),
                              tableN[tableN[id].r].f,Biddy_NOT(tableN[tableN[id].l].f));
    }

    tableN[id].created = TRUE;
  }

}

/**Function****************************************************************
  Synopsis    [Function Bddscout_ReadBDD.]
  Description []
  SideEffects []
  SeeAlso     []
  ************************************************************************/

static Biddy_Edge ReadBDD_R(Biddy_String bdd, int *i, Biddy_String *ch);

Biddy_String
Bddscout_ReadBDD(FILE *s)
{
  Biddy_String bdd;
  int len,n;
  Biddy_String ch;
  int i;
  Biddy_String name;
  Biddy_Edge f;

  len = 0;
  bdd = strdup("");
  do {
    bdd = (Biddy_String) realloc(bdd,len+255);
    n = fread(&bdd[len],1,255,s);
    len = len + n;
  } while (n == 255);

  ch = (Biddy_String) malloc(255);

  i = 0;
  nextCh(bdd,&i,&ch);
  name = strdup(ch);

  f = ReadBDD_R(bdd,&i,&ch);
  if (Biddy_isNull(f)) {
    printf("Bddscout_ReadBDD: ERROR in file: char %d\n",i);
  } else {
    Bddscout_AddFormula(name,f);
  }

  nextCh(bdd,&i,&ch);
  if (strcmp(ch,"")) {
    printf("Bddscout_ReadBDD: ERROR in file: extra characters\n");
  }

  free(ch);

  return(name);
}

static Biddy_Edge
ReadBDD_R(Biddy_String bdd, int *i, Biddy_String *ch)
{
  Biddy_Boolean inv;
  Biddy_String var;
  Biddy_Edge f,n,e,t;

  nextCh(bdd,i,ch);

  if (strcmp(*ch,"*")) {
    inv = FALSE;
  } else {
    inv = TRUE;
    nextCh(bdd,i,ch);
  }

  /*
  printf("<%s>",*ch);
  */

  if (strcmp(*ch,"1")) {
    var = strdup(*ch);
    n = Biddy_FoaTerminal(var);
    nextCh(bdd,i,ch);
    if (strcmp(*ch,"(")) {
      printf("ERROR: <%s>\n",*ch);
      return biddy_null;
    }
    e = ReadBDD_R(bdd,i,ch);
    nextCh(bdd,i,ch);
    if (strcmp(*ch,")")) {
      printf("ERROR: <%s>\n",*ch);
      return biddy_null;
    }
    nextCh(bdd,i,ch);
    if (strcmp(*ch,"(")) {
      printf("ERROR: <%s>\n",*ch);
      return biddy_null;
    }
    t = ReadBDD_R(bdd,i,ch);
    nextCh(bdd,i,ch);
    if (strcmp(*ch,")")) {
      printf("ERROR: <%s>\n",*ch);
      return biddy_null;
    }
    f = Biddy_TransferMark(Biddy_ITE(n,t,e),inv);
    free(var);
  } else {
    f = Biddy_TransferMark(biddy_one,inv);
  }

  return f;
}


/**Function****************************************************************
  Synopsis    [Function Bddscout_WriteBDD.]
  Description []
  SideEffects []
  SeeAlso     []
  ************************************************************************/

static void WriteBDD_R(FILE *funfile, Biddy_Edge f, int *line);

void
Bddscout_WriteBDD(FILE *s, Biddy_String name)
{
  Biddy_Edge f;
  int line;

  if (!Bddscout_FindFormula(name,&f)) {
    printf("Function %s does not exists!\n",name);
    return;
  } else {

    fprintf(s,"%s ",name);

    if (Biddy_isNull(f)) {
      fprintf(s,"NULL\n");
    } else {
      line = strlen(name)+1;
      WriteBDD_R(s,f,&line);
      if (line != 0) fprintf(s,"\n");
    }

  }
}

static void
WriteBDD_R(FILE *funfile, Biddy_Edge f, int *line)
{
  Biddy_String var;

  if (Biddy_GetMark(f)) {
    fprintf(funfile,"* ");
  }

  *line = *line + 2;

  var = Biddy_GetVariableName(f);
  fprintf(funfile,"%s",var);

  *line = *line + strlen(var);

  if (!Biddy_isTerminal(f)) {
    if (*line >= 80) {
      fprintf(funfile,"\n");
      *line = 0;
    }
    fprintf(funfile," (");
    *line = *line + 2;
    WriteBDD_R(funfile,Biddy_GetElse(f),line);
    if (*line >= 80) {
      fprintf(funfile,"\n");
      *line = 0;
    }
    fprintf(funfile,") (");
    *line = *line + 3;
    WriteBDD_R(funfile,Biddy_GetThen(f),line);
    if (*line >= 80) {
      fprintf(funfile,"\n");
      *line = 0;
    }
    fprintf(funfile,")");
    *line = *line + 1;
  }
}

/*-----------------------------------------------------------------------*/
/* TCL related functions                                                 */
/*-----------------------------------------------------------------------*/

/**Function****************************************************************
  Synopsis    [Function Bddscout_Init.]
  Description [This function is used by Tcl/Tk only.]
  SideEffects []
  SeeAlso     []
  ************************************************************************/

static int BddscoutInitPkgCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutExitPkgCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutAboutPkgCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutListFormulaeByNameCmd(
                          ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutListFormulaeByNodeNumberCmd(
                          ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutListFormulaeByNodeMaxLevelCmd(
                          ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutListFormulaeByNodeAvgLevelCmd(
                          ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutWriteBDDviewCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutConstructBDDCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutReadBDDCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutReadBFCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BddscoutWriteBDDCmd(ClientData clientData, Tcl_Interp *interp,
                          int argc, USECONST char **argv);

static int BiddyTableSizeCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableMaxCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableNumberCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableFortifiedCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableFoaCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableCompareCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableAddCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyGarbageNumberCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyTableGeneratedCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyBlockNumberCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyListUsedCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyListMaxLengthCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyListAvgLengthCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyIteCacheFindCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

static int BiddyIteCacheOverwriteCmd(ClientData clientData, Tcl_Interp *interp,
                                int argc, USECONST char **argv);

#ifdef __cplusplus
extern "C" {
#endif

int
Bddscout_Init(Tcl_Interp *interp)
{

#ifdef USE_TCL_STUBS
  if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
    return TCL_ERROR;
  }
#endif

  Tcl_CreateCommand(interp, "bddscout_initPkg", BddscoutInitPkgCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_exitPkg", BddscoutExitPkgCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_aboutPkg", BddscoutAboutPkgCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_listFormulaeByName",
                     BddscoutListFormulaeByNameCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_listFormulaeByNodeNumber",
                     BddscoutListFormulaeByNodeNumberCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_listFormulaeByNodeMaxLevel",
                     BddscoutListFormulaeByNodeMaxLevelCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_listFormulaeByNodeAvgLevel",
                     BddscoutListFormulaeByNodeAvgLevelCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_writeBDDview", BddscoutWriteBDDviewCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_constructBDD", BddscoutConstructBDDCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_readBDD", BddscoutReadBDDCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_readBF", BddscoutReadBFCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "bddscout_writeBDD", BddscoutWriteBDDCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_size", BiddyTableSizeCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_max", BiddyTableMaxCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_number", BiddyTableNumberCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_fortified", BiddyTableFortifiedCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_foa", BiddyTableFoaCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_compare", BiddyTableCompareCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_add", BiddyTableAddCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_garbage_number", BiddyGarbageNumberCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_table_generated", BiddyTableGeneratedCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_block_number", BiddyBlockNumberCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_list_used", BiddyListUsedCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_list_max_length", BiddyListMaxLengthCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_list_avg_length", BiddyListAvgLengthCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_cache_ITE_find", BiddyIteCacheFindCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

  Tcl_CreateCommand(interp, "biddy_cache_ITE_overwrite", BiddyIteCacheOverwriteCmd,
                     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);

#ifdef USE_TCL_STUBS
  return Tcl_PkgProvideEx(interp, "bddscout-lib", "1.0", &bddscoutStubs);
#else
  return Tcl_PkgProvide(interp, "bddscout-lib", "1.0");
#endif
}

#ifdef __cplusplus
}
#endif

static int
BddscoutInitPkgCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                   USECONST char **argv)
{
  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  Biddy_Init();
  bddscoutFormulaeTree = NULL;

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutExitPkgCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                   USECONST char **argv)
{
  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  Bddscout_DeleteFormulaeTree(bddscoutFormulaeTree);
  Biddy_Exit();

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutAboutPkgCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                    USECONST char **argv)
{
  char *bv;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  printf("This is BDD Scout library");
  bv = Biddy_About();
  printf("Bdd package (Biddy v%s)",bv);

  free(bv);

  printf("\n");
  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutListFormulaeByNameCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                    USECONST char **argv)
{
  Biddy_String list;

  list = strdup("");

  Bddscout_ListFormulaByName(&list, bddscoutFormulaeTree);

  Tcl_SetResult(interp, list, TCL_VOLATILE);
  free(list);

  return TCL_OK;
}

static int
BddscoutListFormulaeByNodeNumberCmd(ClientData clientData,
             Tcl_Interp *interp, int argc, USECONST char **argv)
{
  Biddy_String list;

  list = strdup("NULL"); /* required! */

  Bddscout_ListFormulaByNodeNumber(&list, bddscoutFormulaeTree);

  Tcl_SetResult(interp, list, TCL_VOLATILE);
  free(list);

  return TCL_OK;
}

static int
BddscoutListFormulaeByNodeMaxLevelCmd(ClientData clientData,
             Tcl_Interp *interp, int argc, USECONST char **argv)
{
  Biddy_String list;

  list = strdup("NULL"); /* required! */

  Bddscout_ListFormulaByNodeMaxLevel(&list, bddscoutFormulaeTree);

  Tcl_SetResult(interp, list, TCL_VOLATILE);
  free(list);

  return TCL_OK;
}

static int
BddscoutListFormulaeByNodeAvgLevelCmd(ClientData clientData,
             Tcl_Interp *interp, int argc, USECONST char **argv)
{
  Biddy_String list;

  list = strdup("NULL"); /* required! */

  Bddscout_ListFormulaByNodeAvgLevel(&list, bddscoutFormulaeTree);

  Tcl_SetResult(interp, list, TCL_VOLATILE);
  free(list);

  return TCL_OK;
}

static int
BddscoutWriteBDDviewCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String s1,s2,s3;
  FILE *funfile;
  int ERR;

  if (argc != 4) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  s1 = strdup(argv[1]);
  s2 = strdup(argv[2]);
  s3 = strdup(argv[3]);

  funfile = fopen(s2,"w");
  if (!funfile) {
    printf("BddscoutWriteBDDviewCmd: File error (%s)!\n",s2);
    free(s1);
    free(s2);
    free(s3);
    return TCL_ERROR;
  }

  ERR = Bddscout_WriteBDDView(funfile,s3,s1,-1);
  fclose(funfile);

  free(s1);
  free(s2);
  free(s3);

  if (ERR) {
    printf("Bddscout_WriteBDDView error\n");
  }

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutConstructBDDCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String s1,s2,s3,s4;
  int numV,numN;

  if (argc != 5) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  s1 = strdup(argv[1]); /* number of variables */
  s2 = strdup(argv[2]); /* list of variables */
  s3 = strdup(argv[3]); /* number of nodes */
  s4 = strdup(argv[4]); /* BDD */

  numV = atoi(s1);
  numN = atoi(s3);

  Bddscout_ConstructBDD(numV,s2,numN,s4);

  free(s1);
  free(s2);
  free(s3);
  free(s4);

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutWriteBDDCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String s1,s2;
  FILE *funfile;

  if (argc != 3) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  s1 = strdup(argv[1]);
  s2 = strdup(argv[2]);

  funfile = fopen(s1,"w");
  if (!funfile) {
    printf("BddscoutWriteBDDCmd: File error (%s)!\n",s1);
    free(s1);
    free(s2);
    return TCL_ERROR;
  }

  Bddscout_WriteBDD(funfile,s2);

  fclose(funfile);

  free(s1);
  free(s2);

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BddscoutReadBDDCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String s1;
  FILE *funfile;
  Biddy_String name;

  if (argc != 2) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  s1 = strdup(argv[1]);

  funfile = fopen(s1,"r");
  if (!funfile) {
    printf("BddscoutReadBDDCmd: File error (%s)!\n",s1);
    free(s1);
    return TCL_ERROR;
  }

  name = Bddscout_ReadBDD(funfile);

  fclose(funfile);

  free(s1);

  Tcl_SetResult(interp, name, TCL_VOLATILE);
  free(name);

  return TCL_OK;
}

static int
BddscoutReadBFCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String s1;
  FILE *funfile;

  if (argc != 2) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  s1 = strdup(argv[1]);

  funfile = fopen(s1,"r");
  if (!funfile) {
    printf("BddscoutReadBFCmd: File error (%s)!\n",s1);
    free(s1);
    return TCL_ERROR;
  }

  Bddscout_ParseFile(funfile);

  fclose(funfile);

  free(s1);

  Tcl_SetResult(interp, (char *) "", TCL_STATIC);
  return TCL_OK;
}

static int
BiddyTableSizeCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableSize());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableMaxCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableMax());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableNumberCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableNum());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableFortifiedCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableNumF());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableFoaCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%lu",Biddy_NodeTableFOA());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableCompareCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%lu",Biddy_NodeTableCompare());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableAddCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%lu",Biddy_NodeTableAdd());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyGarbageNumberCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableGarbage());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyTableGeneratedCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableGenerated());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyBlockNumberCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_NodeTableBlockNumber());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyListUsedCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_ListUsed());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyListMaxLengthCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%d",Biddy_ListMaxLength());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyListAvgLengthCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%.2f",Biddy_ListAvgLength());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyIteCacheFindCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%lu",Biddy_IteCacheFind());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}

static int
BiddyIteCacheOverwriteCmd(ClientData clientData, Tcl_Interp *interp, int argc,
                  USECONST char **argv)
{
  Biddy_String sup;

  if (argc != 1) {
    Tcl_SetResult(interp, (char *) "wrong # args", TCL_STATIC);
    return TCL_ERROR;
  }

  sup = (Biddy_String) malloc(127);
  sprintf(sup,"%lu",Biddy_IteCacheOverwrite());
  Tcl_SetResult(interp, sup, TCL_VOLATILE);
  free(sup);

  return TCL_OK;
}
