// Copyright (c) 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef TOOLS_GN_NINJA_TARGET_WRITER_H_
#define TOOLS_GN_NINJA_TARGET_WRITER_H_

#include <iosfwd>

#include "base/macros.h"
#include "tools/gn/path_output.h"
#include "tools/gn/substitution_type.h"

class FileTemplate;
class OutputFile;
class Settings;
class Target;
struct SubstitutionBits;

// Generates one target's ".ninja" file. The toplevel "build.ninja" file is
// generated by the NinjaBuildWriter.
class NinjaTargetWriter {
 public:
  NinjaTargetWriter(const Target* target, std::ostream& out);
  virtual ~NinjaTargetWriter();

  // Returns the build line to be written to the toolchain build file.
  //
  // Some targets have their rules written to separate files, and some can have
  // their rules coalesced in the main build file. For the coalesced case, this
  // function will return the rules as a string. For the separate file case,
  // the separate ninja file will be written and the return string will be the
  // subninja command to load that file.
  static std::string RunAndWriteFile(const Target* target);

  virtual void Run() = 0;

 protected:
  // Writes out the substitution values that are shared between the different
  // types of tools (target gen dir, target label, etc.). Only the substitutions
  // identified by the given bits will be written.
  void WriteSharedVars(const SubstitutionBits& bits);

  // Writes to the output stream a stamp rule for input dependencies, and
  // returns the file to be appended to source rules that encodes the
  // order-only dependencies for the current target. The returned OutputFile
  // will be empty if there are no implicit dependencies and no extra target
  // dependencies passed in.
  OutputFile WriteInputDepsStampAndGetDep(
      const std::vector<const Target*>& extra_hard_deps) const;

  // Writes to the output file a final stamp rule for the target that stamps
  // the given list of files. This function assumes the stamp is for the target
  // as a whole so the stamp file is set as the target's dependency output.
  void WriteStampForTarget(const std::vector<OutputFile>& deps,
                           const std::vector<OutputFile>& order_only_deps);

  const Settings* settings_;  // Non-owning.
  const Target* target_;  // Non-owning.
  std::ostream& out_;
  PathOutput path_output_;

 private:
  void WriteCopyRules();
  void WriteEscapedSubstitution(SubstitutionType type);

  DISALLOW_COPY_AND_ASSIGN(NinjaTargetWriter);
};

#endif  // TOOLS_GN_NINJA_TARGET_WRITER_H_
