#!/usr/bin/python
# Copyright (c) 2014 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Utility to rename a config in some subset of our GM expectation files.

Created for http://skbug.com/2752 ('split existing "gpu" GM results into "gl"
and "gles"')

Run with -h to see usage.

Example command lines:
  rename_config.py gpu gles '.*Android.*'

TODO(epoger): Once https://codereview.chromium.org/397103003/ is committed,
we should add a unittest.  Until then, we can test this as follows:

OLD=expectations/gm && NEW=/tmp/expectations && \
  rm -rf $NEW && \
  cp -a $OLD $NEW && \
  gm/rename_config.py msaa4 gles-msaa4 '.*Android.*' \
    --expectations-root $NEW && \
  diff --recursive $OLD $NEW
"""
__author__ = 'Elliot Poger'

import argparse
import os
import re

import gm_json

DEFAULT_EXPECTATIONS_ROOT = os.path.join(
    os.path.dirname(__file__), os.pardir, 'expectations', 'gm')
IMAGE_FILENAME_RE = re.compile(gm_json.IMAGE_FILENAME_PATTERN)


class Renamer(object):

  def __init__(self, args):
    """
    Params:
      args: the Namespace object generated by argparse.parse_args()
    """
    self._args = args

  def run(self):
    """Perform all the subsitutions."""
    for path in self._get_file_list():
      self._rename_config(path=path,
                          old=self._args.old_config_name,
                          new=self._args.new_config_name)

  def _rename_config(self, path, old, new):
    """Renames all instances of a config within a GM expectations file.

    Params:
      path: path to file which will be modified in place
      old: old config name
      new: new config name
    """
    dic = gm_json.LoadFromFile(file_path=path)
    expected_results = dic[gm_json.JSONKEY_EXPECTEDRESULTS]
    orig_keys = expected_results.keys()
    for key in orig_keys:
      result = expected_results.pop(key)
      (testname, config) = IMAGE_FILENAME_RE.match(key).groups()
      if config == old:
        config = new
      key = '%s_%s.png' % (testname, config)
      expected_results[key] = result
    gm_json.WriteToFile(json_dict=dic, file_path=path)

  def _get_file_list(self):
    """Returns the list of files we want to operate on (the complete path
    to each file)."""
    root = self._args.expectations_root
    regex = re.compile(self._args.builder_name_pattern)
    return [os.path.join(root, builder, 'expected-results.json')
            for builder in os.listdir(root)
            if regex.match(builder)]


def main():
  parser = argparse.ArgumentParser()
  parser.add_argument('old_config_name',
                      help=('Config name we want to replace.'))
  parser.add_argument('new_config_name',
                      help=('Config name we want to replace the old one with.'))
  parser.add_argument('builder_name_pattern',
                      help=('Regex pattern describing which builders we want '
                            'to make the substitution for; \'.*\' to perform '
                            'the replacement on all builders.'))
  parser.add_argument('--expectations-root',
                      default=DEFAULT_EXPECTATIONS_ROOT,
                      help=('Root of the GM expectations dir; defaults to '
                            '%(default)s'))
  args = parser.parse_args()
  renamer = Renamer(args)
  renamer.run()

if __name__ == '__main__':
  main()
