/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the config.tests of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qwaylandintegration_p.h"

#include "qwaylanddisplay_p.h"
#include "qwaylandshmwindow_p.h"
#include "qwaylandinputdevice_p.h"
#include "qwaylandinputcontext_p.h"
#include "qwaylandinputmethodcontext_p.h"
#include "qwaylandshmbackingstore_p.h"
#include "qwaylandnativeinterface_p.h"
#if QT_CONFIG(clipboard)
#include "qwaylandclipboard_p.h"
#endif
#include "qwaylanddnd_p.h"
#include "qwaylandwindowmanagerintegration_p.h"
#include "qwaylandscreen_p.h"
#include "qwaylandcursor_p.h"

#if defined(Q_OS_MACOS)
#  include <QtGui/private/qcoretextfontdatabase_p.h>
#  include <QtGui/private/qfontengine_coretext_p.h>
#else
#  include <QtGui/private/qgenericunixfontdatabase_p.h>
#endif
#include <QtGui/private/qgenericunixeventdispatcher_p.h>
#include <QtGui/private/qgenericunixthemes_p.h>

#include <QtGui/private/qguiapplication_p.h>

#include <qpa/qwindowsysteminterface.h>
#include <qpa/qplatformcursor.h>
#include <QtGui/QSurfaceFormat>
#if QT_CONFIG(opengl)
#include <QtGui/QOpenGLContext>
#endif // QT_CONFIG(opengl)
#include <QSocketNotifier>

#include <qpa/qplatforminputcontextfactory_p.h>
#include <qpa/qplatformaccessibility.h>
#include <qpa/qplatforminputcontext.h>

#include "qwaylandhardwareintegration_p.h"
#include "qwaylandclientbufferintegration_p.h"
#include "qwaylandclientbufferintegrationfactory_p.h"

#include "qwaylandserverbufferintegration_p.h"
#include "qwaylandserverbufferintegrationfactory_p.h"
#include "qwaylandshellsurface_p.h"

#include "qwaylandshellintegration_p.h"
#include "qwaylandshellintegrationfactory_p.h"

#include "qwaylandinputdeviceintegration_p.h"
#include "qwaylandinputdeviceintegrationfactory_p.h"
#include "qwaylandwindow_p.h"

#if QT_CONFIG(accessibility_atspi_bridge)
#include <QtGui/private/qspiaccessiblebridge_p.h>
#endif

#if QT_CONFIG(xkbcommon)
#include <QtGui/private/qxkbcommon_p.h>
#endif

#if QT_CONFIG(vulkan)
#include "qwaylandvulkaninstance_p.h"
#include "qwaylandvulkanwindow_p.h"
#endif

QT_BEGIN_NAMESPACE

namespace QtWaylandClient {

QWaylandIntegration::QWaylandIntegration()
#if defined(Q_OS_MACOS)
    : mFontDb(new QCoreTextFontDatabaseEngineFactory<QCoreTextFontEngine>)
#else
    : mFontDb(new QGenericUnixFontDatabase())
#endif
{
    mDisplay.reset(new QWaylandDisplay(this));
    if (!mDisplay->isInitialized()) {
        mFailed = true;
        return;
    }

    // ### Not ideal...
    // We don't want to use QPlatformWindow::requestActivate here, since that gives a warning
    // for most shells. Also, we don't want to put this into the specific shells that can use
    // it, since we want to support more than one shell in one client.
    // In addition, this will send a new requestActivate when the focus object changes, even if
    // the focus window stays the same.
    QObject::connect(qApp, &QGuiApplication::focusObjectChanged, qApp, [](){
        QWindow *fw = QGuiApplication::focusWindow();
        auto *w = fw ? static_cast<QWaylandWindow*>(fw->handle()) : nullptr;
        if (w && w->shellSurface())
            w->shellSurface()->requestActivate();
    });
}

QWaylandIntegration::~QWaylandIntegration()
{
}

QPlatformNativeInterface * QWaylandIntegration::nativeInterface() const
{
    return mNativeInterface.data();
}

bool QWaylandIntegration::hasCapability(QPlatformIntegration::Capability cap) const
{
    switch (cap) {
    case ThreadedPixmaps: return true;
    case OpenGL:
        return mDisplay->clientBufferIntegration();
    case ThreadedOpenGL:
        return mDisplay->clientBufferIntegration() && mDisplay->clientBufferIntegration()->supportsThreadedOpenGL();
    case BufferQueueingOpenGL:
        return true;
    case MultipleWindows:
    case NonFullScreenWindows:
        return true;
    case RasterGLSurface:
        return true;
    case WindowActivation:
        return false;
    default: return QPlatformIntegration::hasCapability(cap);
    }
}

QPlatformWindow *QWaylandIntegration::createPlatformWindow(QWindow *window) const
{
    if ((window->surfaceType() == QWindow::OpenGLSurface || window->surfaceType() == QWindow::RasterGLSurface)
        && mDisplay->clientBufferIntegration())
        return mDisplay->clientBufferIntegration()->createEglWindow(window);

#if QT_CONFIG(vulkan)
    if (window->surfaceType() == QSurface::VulkanSurface)
        return new QWaylandVulkanWindow(window, mDisplay.data());
#endif // QT_CONFIG(vulkan)

    return new QWaylandShmWindow(window, mDisplay.data());
}

#if QT_CONFIG(opengl)
QPlatformOpenGLContext *QWaylandIntegration::createPlatformOpenGLContext(QOpenGLContext *context) const
{
    if (mDisplay->clientBufferIntegration())
        return mDisplay->clientBufferIntegration()->createPlatformOpenGLContext(context->format(), context->shareHandle());
    return nullptr;
}
#endif  // opengl

QPlatformBackingStore *QWaylandIntegration::createPlatformBackingStore(QWindow *window) const
{
    return new QWaylandShmBackingStore(window, mDisplay.data());
}

QAbstractEventDispatcher *QWaylandIntegration::createEventDispatcher() const
{
    return createUnixEventDispatcher();
}

QPlatformNativeInterface *QWaylandIntegration::createPlatformNativeInterface()
{
    return new QWaylandNativeInterface(this);
}

// Support platform specific initialization
void QWaylandIntegration::initializePlatform()
{
    mDisplay->initialize();

    mNativeInterface.reset(createPlatformNativeInterface());
    initializeInputDeviceIntegration();
#if QT_CONFIG(clipboard)
    mClipboard.reset(new QWaylandClipboard(mDisplay.data()));
#endif
#if QT_CONFIG(draganddrop)
    mDrag.reset(new QWaylandDrag(mDisplay.data()));
#endif

    reconfigureInputContext();
}

void QWaylandIntegration::initialize()
{
    mDisplay->initEventThread();

    // Call this after initializing event thread for QWaylandDisplay::forceRoundTrip()
    initializePlatform();

    // But the aboutToBlock() and awake() should be connected after initializePlatform().
    // Otherwise the connected flushRequests() may consumes up all events before processEvents starts to wait,
    // so that processEvents(QEventLoop::WaitForMoreEvents) may be blocked in the forceRoundTrip().
    QAbstractEventDispatcher *dispatcher = QGuiApplicationPrivate::eventDispatcher;
    QObject::connect(dispatcher, SIGNAL(aboutToBlock()), mDisplay.data(), SLOT(flushRequests()));
    QObject::connect(dispatcher, SIGNAL(awake()), mDisplay.data(), SLOT(flushRequests()));

    // Qt does not support running with no screens
    mDisplay->ensureScreen();
}

QPlatformFontDatabase *QWaylandIntegration::fontDatabase() const
{
    return mFontDb.data();
}

#if QT_CONFIG(clipboard)
QPlatformClipboard *QWaylandIntegration::clipboard() const
{
    return mClipboard.data();
}
#endif

#if QT_CONFIG(draganddrop)
QPlatformDrag *QWaylandIntegration::drag() const
{
    return mDrag.data();
}
#endif  // draganddrop

QPlatformInputContext *QWaylandIntegration::inputContext() const
{
    return mInputContext.data();
}

QVariant QWaylandIntegration::styleHint(StyleHint hint) const
{
    if (hint == ShowIsFullScreen && mDisplay->windowManagerIntegration())
        return mDisplay->windowManagerIntegration()->showIsFullScreen();

    return QPlatformIntegration::styleHint(hint);
}

#if QT_CONFIG(accessibility)
QPlatformAccessibility *QWaylandIntegration::accessibility() const
{
    if (!mAccessibility) {
#if QT_CONFIG(accessibility_atspi_bridge)
        Q_ASSERT_X(QCoreApplication::eventDispatcher(), "QWaylandIntegration",
            "Initializing accessibility without event-dispatcher!");
        mAccessibility.reset(new QSpiAccessibleBridge());
#else
        mAccessibility.reset(new QPlatformAccessibility());
#endif
    }
    return mAccessibility.data();
}
#endif

QPlatformServices *QWaylandIntegration::services() const
{
    return mDisplay->windowManagerIntegration();
}

QWaylandDisplay *QWaylandIntegration::display() const
{
    return mDisplay.data();
}

Qt::KeyboardModifiers QWaylandIntegration::queryKeyboardModifiers() const
{
    if (auto *seat = mDisplay->currentInputDevice(); seat && seat->keyboardFocus()) {
        return seat->modifiers();
    }
    return Qt::NoModifier;
}

QList<int> QWaylandIntegration::possibleKeys(const QKeyEvent *event) const
{
    if (auto *seat = mDisplay->currentInputDevice())
        return seat->possibleKeys(event);
    return {};
}

QStringList QWaylandIntegration::themeNames() const
{
    return QGenericUnixTheme::themeNames();
}

QPlatformTheme *QWaylandIntegration::createPlatformTheme(const QString &name) const
{
    return QGenericUnixTheme::createUnixTheme(name);
}

QWaylandScreen *QWaylandIntegration::createPlatformScreen(QWaylandDisplay *waylandDisplay, int version, uint32_t id) const
{
   return new QWaylandScreen(waylandDisplay, version, id);
}

QWaylandCursor *QWaylandIntegration::createPlatformCursor(QWaylandDisplay *display) const
{
   return new QWaylandCursor(display);
}

#if QT_CONFIG(vulkan)
QPlatformVulkanInstance *QWaylandIntegration::createPlatformVulkanInstance(QVulkanInstance *instance) const
{
    return new QWaylandVulkanInstance(instance);
}
#endif // QT_CONFIG(vulkan)

// May be called from non-GUI threads
QWaylandClientBufferIntegration *QWaylandIntegration::clientBufferIntegration() const
{
    // Do an inexpensive check first to avoid locking whenever possible
    if (Q_UNLIKELY(!mClientBufferIntegrationInitialized))
        const_cast<QWaylandIntegration *>(this)->initializeClientBufferIntegration();

    Q_ASSERT(mClientBufferIntegrationInitialized);
    return mClientBufferIntegration && mClientBufferIntegration->isValid() ? mClientBufferIntegration.data() : nullptr;
}

QWaylandServerBufferIntegration *QWaylandIntegration::serverBufferIntegration() const
{
    if (!mServerBufferIntegrationInitialized)
        const_cast<QWaylandIntegration *>(this)->initializeServerBufferIntegration();

    return mServerBufferIntegration.data();
}

QWaylandShellIntegration *QWaylandIntegration::shellIntegration() const
{
    if (!mShellIntegrationInitialized)
        const_cast<QWaylandIntegration *>(this)->initializeShellIntegration();

    return mShellIntegration.data();
}

// May be called from non-GUI threads
void QWaylandIntegration::initializeClientBufferIntegration()
{
    QMutexLocker lock(&mClientBufferInitLock);
    if (mClientBufferIntegrationInitialized)
        return;

    QString targetKey = QString::fromLocal8Bit(qgetenv("QT_WAYLAND_CLIENT_BUFFER_INTEGRATION"));

    if (targetKey.isEmpty()) {
        if (mDisplay->hardwareIntegration()
                && mDisplay->hardwareIntegration()->clientBufferIntegration() != QLatin1String("wayland-eglstream-controller")
                && mDisplay->hardwareIntegration()->clientBufferIntegration() != QLatin1String("linux-dmabuf-unstable-v1")) {
            targetKey = mDisplay->hardwareIntegration()->clientBufferIntegration();
        } else {
            targetKey = QLatin1String("wayland-egl");
        }
    }

    if (targetKey.isEmpty()) {
        qWarning("Failed to determine what client buffer integration to use");
    } else {
        QStringList keys = QWaylandClientBufferIntegrationFactory::keys();
        qCDebug(lcQpaWayland) << "Available client buffer integrations:" << keys;

        if (keys.contains(targetKey))
            mClientBufferIntegration.reset(QWaylandClientBufferIntegrationFactory::create(targetKey, QStringList()));

        if (mClientBufferIntegration) {
            qCDebug(lcQpaWayland) << "Initializing client buffer integration" << targetKey;
            mClientBufferIntegration->initialize(mDisplay.data());
        } else {
            qCWarning(lcQpaWayland) << "Failed to load client buffer integration:" << targetKey;
            qCWarning(lcQpaWayland) << "Available client buffer integrations:" << keys;
        }
    }

    // This must be set last to make sure other threads don't use the
    // integration before initialization is complete.
    mClientBufferIntegrationInitialized = true;
}

void QWaylandIntegration::initializeServerBufferIntegration()
{
    mServerBufferIntegrationInitialized = true;

    QString targetKey = QString::fromLocal8Bit(qgetenv("QT_WAYLAND_SERVER_BUFFER_INTEGRATION"));

    if (targetKey.isEmpty() && mDisplay->hardwareIntegration())
        targetKey = mDisplay->hardwareIntegration()->serverBufferIntegration();

    if (targetKey.isEmpty()) {
        qWarning("Failed to determine what server buffer integration to use");
        return;
    }

    QStringList keys = QWaylandServerBufferIntegrationFactory::keys();
    qCDebug(lcQpaWayland) << "Available server buffer integrations:" << keys;

    if (keys.contains(targetKey))
        mServerBufferIntegration.reset(QWaylandServerBufferIntegrationFactory::create(targetKey, QStringList()));

    if (mServerBufferIntegration) {
        qCDebug(lcQpaWayland) << "Initializing server buffer integration" << targetKey;
        mServerBufferIntegration->initialize(mDisplay.data());
    } else {
        qCWarning(lcQpaWayland) << "Failed to load server buffer integration: " <<  targetKey;
        qCWarning(lcQpaWayland) << "Available server buffer integrations:" << keys;
    }
}

void QWaylandIntegration::initializeShellIntegration()
{
    mShellIntegrationInitialized = true;

    QByteArray integrationNames = qgetenv("QT_WAYLAND_SHELL_INTEGRATION");
    QString targetKeys = QString::fromLocal8Bit(integrationNames);

    QStringList preferredShells;
    if (!targetKeys.isEmpty()) {
        preferredShells = targetKeys.split(QLatin1Char(';'));
    } else {
        preferredShells << QLatin1String("xdg-shell");
        preferredShells << QLatin1String("wl-shell") << QLatin1String("ivi-shell");
        preferredShells << QLatin1String("qt-shell");
    }

    for (const QString &preferredShell : qAsConst(preferredShells)) {
        mShellIntegration.reset(createShellIntegration(preferredShell));
        if (mShellIntegration) {
            qCDebug(lcQpaWayland, "Using the '%s' shell integration", qPrintable(preferredShell));
            break;
        }
    }

    if (!mShellIntegration) {
        qCWarning(lcQpaWayland) << "Loading shell integration failed.";
        qCWarning(lcQpaWayland) << "Attempted to load the following shells" << preferredShells;
    }

    QWindowSystemInterfacePrivate::TabletEvent::setPlatformSynthesizesMouse(false);
}

QWaylandInputDevice *QWaylandIntegration::createInputDevice(QWaylandDisplay *display, int version, uint32_t id) const
{
    if (mInputDeviceIntegration) {
        return mInputDeviceIntegration->createInputDevice(display, version, id);
    }
    return new QWaylandInputDevice(display, version, id);
}

void QWaylandIntegration::initializeInputDeviceIntegration()
{
    QByteArray integrationName = qgetenv("QT_WAYLAND_INPUTDEVICE_INTEGRATION");
    QString targetKey = QString::fromLocal8Bit(integrationName);

    if (targetKey.isEmpty()) {
        return;
    }

    QStringList keys = QWaylandInputDeviceIntegrationFactory::keys();
    if (keys.contains(targetKey)) {
        mInputDeviceIntegration.reset(QWaylandInputDeviceIntegrationFactory::create(targetKey, QStringList()));
        qDebug("Using the '%s' input device integration", qPrintable(targetKey));
    } else {
        qWarning("Wayland inputdevice integration '%s' not found, using default", qPrintable(targetKey));
    }
}

void QWaylandIntegration::reconfigureInputContext()
{
    if (!mDisplay) {
        // This function can be called from QWaylandDisplay::registry_global() when we
        // are in process of constructing QWaylandDisplay. Configuring input context
        // in that case is done by calling reconfigureInputContext() from QWaylandIntegration
        // constructor, after QWaylandDisplay has been constructed.
        return;
    }

    const QString &requested = QPlatformInputContextFactory::requested();
    if (requested == QLatin1String("qtvirtualkeyboard"))
        qCWarning(lcQpaWayland) << "qtvirtualkeyboard currently is not supported at client-side,"
                                   " use QT_IM_MODULE=qtvirtualkeyboard at compositor-side.";

    if (requested.isNull()) {
        if (mDisplay->textInputMethodManager() != nullptr)
            mInputContext.reset(new QWaylandInputMethodContext(mDisplay.data()));
#if QT_WAYLAND_TEXT_INPUT_V4_WIP
        else if (mDisplay->textInputManagerv2() != nullptr || mDisplay->textInputManagerv4() != nullptr)
#else //  QT_WAYLAND_TEXT_INPUT_V4_WIP
        else if (mDisplay->textInputManagerv2() != nullptr)
#endif // QT_WAYLAND_TEXT_INPUT_V4_WIP
            mInputContext.reset(new QWaylandInputContext(mDisplay.data()));
    } else {
        mInputContext.reset(QPlatformInputContextFactory::create(requested));
    }

    const QString defaultInputContext(QStringLiteral("compose"));
    if ((!mInputContext || !mInputContext->isValid()) && requested != defaultInputContext)
        mInputContext.reset(QPlatformInputContextFactory::create(defaultInputContext));

#if QT_CONFIG(xkbcommon)
    QXkbCommon::setXkbContext(mInputContext.data(), mDisplay->xkbContext());
    if (QWaylandInputContext* waylandInput = qobject_cast<QWaylandInputContext*>(mInputContext.get())) {
        waylandInput->setXkbContext(mDisplay->xkbContext());
    }
#endif

    qCDebug(lcQpaWayland) << "using input method:" << (inputContext() ? inputContext()->metaObject()->className() : "<none>");
}

QWaylandShellIntegration *QWaylandIntegration::createShellIntegration(const QString &integrationName)
{
    if (QWaylandShellIntegrationFactory::keys().contains(integrationName)) {
        return QWaylandShellIntegrationFactory::create(integrationName, mDisplay.data());
    } else {
        qCWarning(lcQpaWayland) << "No shell integration named" << integrationName << "found";
        return nullptr;
    }
}

}

QT_END_NAMESPACE
