/*
 *  Copyright 2017 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

#ifndef WEBRTC_API_ORTC_RTPTRANSPORTINTERFACE_H_
#define WEBRTC_API_ORTC_RTPTRANSPORTINTERFACE_H_

#include <string>

#include "webrtc/api/ortc/packettransportinterface.h"
#include "webrtc/api/rtcerror.h"
#include "webrtc/rtc_base/optional.h"

namespace webrtc {

class RtpTransportAdapter;

struct RtcpParameters {
  // The SSRC to be used in the "SSRC of packet sender" field. If not set, one
  // will be chosen by the implementation.
  // TODO(deadbeef): Not implemented.
  rtc::Optional<uint32_t> ssrc;

  // The Canonical Name (CNAME) used by RTCP (e.g. in SDES messages).
  //
  // If empty in the construction of the RtpTransport, one will be generated by
  // the implementation, and returned in GetRtcpParameters. Multiple
  // RtpTransports created by the same OrtcFactory will use the same generated
  // CNAME.
  //
  // If empty when passed into SetRtcpParameters, the CNAME simply won't be
  // modified.
  std::string cname;

  // Send reduced-size RTCP?
  bool reduced_size = false;

  // Send RTCP multiplexed on the RTP transport?
  bool mux = true;

  bool operator==(const RtcpParameters& o) const {
    return ssrc == o.ssrc && cname == o.cname &&
           reduced_size == o.reduced_size && mux == o.mux;
  }
  bool operator!=(const RtcpParameters& o) const { return !(*this == o); }
};

// Base class for different types of RTP transports that can be created by an
// OrtcFactory. Used by RtpSenders/RtpReceivers.
//
// This is not present in the standard ORTC API, but exists here for a few
// reasons. Firstly, it allows different types of RTP transports to be used:
// DTLS-SRTP (which is required for the web), but also SDES-SRTP and
// unencrypted RTP. It also simplifies the handling of RTCP muxing, and
// provides a better API point for it.
//
// Note that Edge's implementation of ORTC provides a similar API point, called
// RTCSrtpSdesTransport:
// https://msdn.microsoft.com/en-us/library/mt502527(v=vs.85).aspx
class RtpTransportInterface {
 public:
  virtual ~RtpTransportInterface() {}

  // Returns packet transport that's used to send RTP packets.
  virtual PacketTransportInterface* GetRtpPacketTransport() const = 0;

  // Returns separate packet transport that's used to send RTCP packets. If
  // RTCP multiplexing is being used, returns null.
  virtual PacketTransportInterface* GetRtcpPacketTransport() const = 0;

  // Set/get RTCP params. Can be used to enable RTCP muxing or reduced-size
  // RTCP if initially not enabled.
  //
  // Changing |mux| from "true" to "false" is not allowed, and changing the
  // CNAME is currently unsupported.
  virtual RTCError SetRtcpParameters(const RtcpParameters& parameters) = 0;
  // Returns last set or constructed-with parameters. If |cname| was empty in
  // construction, the generated CNAME will be present in the returned
  // parameters (see above).
  virtual RtcpParameters GetRtcpParameters() const = 0;

 protected:
  // Only for internal use. Returns a pointer to an internal interface, for use
  // by the implementation.
  virtual RtpTransportAdapter* GetInternal() = 0;

  // Classes that can use this internal interface.
  friend class OrtcFactory;
  friend class OrtcRtpSenderAdapter;
  friend class OrtcRtpReceiverAdapter;
  friend class RtpTransportControllerAdapter;
  friend class RtpTransportAdapter;
};

}  // namespace webrtc

#endif  // WEBRTC_API_ORTC_RTPTRANSPORTINTERFACE_H_
