/*
 * Copyright (C) 2010 Apple Inc. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef SchemeRegistry_h
#define SchemeRegistry_h

#include "platform/PlatformExport.h"
#include "platform/wtf/Allocator.h"
#include "platform/wtf/HashMap.h"
#include "platform/wtf/HashSet.h"
#include "platform/wtf/text/StringHash.h"
#include "platform/wtf/text/WTFString.h"

namespace blink {

using URLSchemesSet = HashSet<String>;

template <typename Mapped, typename MappedTraits>
using URLSchemesMap = HashMap<String,
                              Mapped,
                              DefaultHash<String>::Hash,
                              HashTraits<String>,
                              MappedTraits>;

class PLATFORM_EXPORT SchemeRegistry {
  STATIC_ONLY(SchemeRegistry);

 public:
  static void Initialize();

  static void RegisterURLSchemeAsLocal(const String&);
  static bool ShouldTreatURLSchemeAsLocal(const String&);

  static bool ShouldTreatURLSchemeAsRestrictingMixedContent(const String&);

  // Subresources transported by secure schemes do not trigger mixed content
  // warnings. For example, https and data are secure schemes because they
  // cannot be corrupted by active network attackers.
  static void RegisterURLSchemeAsSecure(const String&);
  static bool ShouldTreatURLSchemeAsSecure(const String&);

  static void RegisterURLSchemeAsNoAccess(const String&);
  static bool ShouldTreatURLSchemeAsNoAccess(const String&);

  // Display-isolated schemes can only be displayed (in the sense of
  // SecurityOrigin::canDisplay) by documents from the same scheme.
  static void RegisterURLSchemeAsDisplayIsolated(const String&);
  static bool ShouldTreatURLSchemeAsDisplayIsolated(const String&);

  static bool ShouldLoadURLSchemeAsEmptyDocument(const String&);

  static void SetDomainRelaxationForbiddenForURLScheme(bool forbidden,
                                                       const String&);
  static bool IsDomainRelaxationForbiddenForURLScheme(const String&);

  // Such schemes should delegate to SecurityOrigin::canRequest for any URL
  // passed to SecurityOrigin::canDisplay.
  static bool CanDisplayOnlyIfCanRequest(const String& scheme);

  // Schemes against which javascript: URLs should not be allowed to run (stop
  // bookmarklets from running on sensitive pages).
  static void RegisterURLSchemeAsNotAllowingJavascriptURLs(
      const String& scheme);
  static bool ShouldTreatURLSchemeAsNotAllowingJavascriptURLs(
      const String& scheme);

  // Allow non-HTTP schemes to be registered to allow CORS requests.
  static void RegisterURLSchemeAsCORSEnabled(const String& scheme);
  static bool ShouldTreatURLSchemeAsCORSEnabled(const String& scheme);

  // Serialize the registered schemes in a comma-separated list.
  static String ListOfCORSEnabledURLSchemes();

  // "Legacy" schemes (e.g. 'ftp:', 'gopher:') which we might want to treat
  // differently from "webby" schemes.
  static bool ShouldTreatURLSchemeAsLegacy(const String& scheme);

  // Does the scheme represent a location relevant to web compatibility metrics?
  static bool ShouldTrackUsageMetricsForScheme(const String& scheme);

  // Schemes that can register a service worker.
  static void RegisterURLSchemeAsAllowingServiceWorkers(const String& scheme);
  static bool ShouldTreatURLSchemeAsAllowingServiceWorkers(
      const String& scheme);

  // HTTP-like schemes that are treated as supporting the Fetch API.
  static void RegisterURLSchemeAsSupportingFetchAPI(const String& scheme);
  static bool ShouldTreatURLSchemeAsSupportingFetchAPI(const String& scheme);

  // Schemes which override the first-/third-party checks on a Document.
  static void RegisterURLSchemeAsFirstPartyWhenTopLevel(const String& scheme);
  static void RemoveURLSchemeAsFirstPartyWhenTopLevel(const String& scheme);
  static bool ShouldTreatURLSchemeAsFirstPartyWhenTopLevel(
      const String& scheme);

  // Schemes that can be used in a referrer.
  static void RegisterURLSchemeAsAllowedForReferrer(const String& scheme);
  static void RemoveURLSchemeAsAllowedForReferrer(const String& scheme);
  static bool ShouldTreatURLSchemeAsAllowedForReferrer(const String& scheme);

  // Allow resources from some schemes to load on a page, regardless of its
  // Content Security Policy.
  enum PolicyAreas : uint32_t {
    kPolicyAreaNone = 0,
    kPolicyAreaImage = 1 << 0,
    kPolicyAreaStyle = 1 << 1,
    // Add more policy areas as needed by clients.
    kPolicyAreaAll = ~static_cast<uint32_t>(0),
  };
  static void RegisterURLSchemeAsBypassingContentSecurityPolicy(
      const String& scheme,
      PolicyAreas = kPolicyAreaAll);
  static void RemoveURLSchemeRegisteredAsBypassingContentSecurityPolicy(
      const String& scheme);
  static bool SchemeShouldBypassContentSecurityPolicy(
      const String& scheme,
      PolicyAreas = kPolicyAreaAll);

  // Schemes which bypass Secure Context checks defined in
  // https://w3c.github.io/webappsec/specs/powerfulfeatures/#is-origin-trustworthy.
  static void RegisterURLSchemeBypassingSecureContextCheck(
      const String& scheme);
  static bool SchemeShouldBypassSecureContextCheck(const String& scheme);

 private:
  static const URLSchemesSet& LocalSchemes();
};

}  // namespace blink

#endif  // SchemeRegistry_h
