/****************************************************************************
**
** Copyright (C) 2016 Paul Lemire
** Copyright (C) 2014 Klaralvdalens Datakonsult AB (KDAB).
** Contact: https://www.qt.io/licensing/
**
** This file is part of the Qt3D module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H
#define QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists for the convenience
// of other Qt classes.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <Qt3DCore/qaspectjob.h>
#include <Qt3DRender/private/renderlogging_p.h>
#include <Qt3DRender/private/job_common_p.h>
#include <QSize>
#include <QElapsedTimer>

QT_BEGIN_NAMESPACE

namespace Qt3DRender {

namespace Render {

class FrameGraphNode;

template<class RenderView, class Renderer>
class RenderViewInitializerJob : public Qt3DCore::QAspectJob
{
public:
    RenderViewInitializerJob()
        : Qt3DCore::QAspectJob()
    {
        SET_JOB_RUN_STAT_TYPE(this, JobTypes::RenderView, RenderViewInitializerJob::renderViewInstanceCounter++)
    }
    ~RenderViewInitializerJob()
    {
        RenderViewInitializerJob::renderViewInstanceCounter--;
    }

    inline void setRenderer(Renderer *renderer) { m_renderer = renderer; }
    inline RenderView *renderView() const noexcept { return m_renderView; }

    inline void setFrameGraphLeafNode(FrameGraphNode *fgLeaf)
    {
        m_fgLeaf = fgLeaf;
    }

    // Sets the position in the queue of RenderViews that the
    // RenderView generated by this job should be inserted. This is
    // used to ensure that for example a RenderView for creating
    // a shadow map texture is submitted before the RenderView that
    // contains commands making use of the shadow map
    inline void setSubmitOrderIndex(int index) { m_index = index; }
    inline int submitOrderIndex() const { return m_index; }

    void run() override
    {
#if defined(QT3D_RENDER_VIEW_JOB_TIMINGS)
        QElapsedTimer timer;
        timer.start();
        qint64 gatherLightsTime;
        qint64 buildCommandsTime;
#endif

        // Create a RenderView object
        m_renderView = new RenderView;

        // RenderView should allocate heap resources using only the currentFrameAllocator
        m_renderView->setRenderer(m_renderer);

        // Populate the renderview's configuration from the framegraph
        RenderView::setRenderViewConfigFromFrameGraphLeafNode(m_renderView, m_fgLeaf);
#if defined(QT3D_RENDER_VIEW_JOB_TIMINGS)
        qint64 gatherStateTime = timer.nsecsElapsed();
        timer.restart();
#endif
    }

private:
    Renderer *m_renderer = nullptr;
    FrameGraphNode *m_fgLeaf = nullptr;
    RenderView *m_renderView = nullptr;
    int m_index = 0;

    static int renderViewInstanceCounter;
};

template<class RenderView, class Renderer>
int RenderViewInitializerJob<RenderView, Renderer>::renderViewInstanceCounter = 0;

template<class RenderView, class Renderer>
using RenderViewInitializerJobPtr = QSharedPointer<RenderViewInitializerJob<RenderView, Renderer>>;

} // namespace Render
} // namespace Qt3DRender

QT_END_NAMESPACE

#endif // QT3DRENDER_RENDER_RENDERVIEWINITIALIZERJOB_H
